<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\PaymentSpedition;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\PaymentSpeditions;

class PaymentSpeditionsGrid extends BaseControl
{
	/** @var PaymentSpeditions */
	protected $paymentSpeditionsService;

	/** var Payments */
	protected $paymentsService;

	/** @var Speditions */
	protected $speditionsService;

	/** @var Sites */
	protected $sitesService;

	/** @var IPaymentSpeditionFormFactory */
	protected $formFactory;

	public function __construct(PaymentSpeditions $paymentSpeditions, Payments $payments, Speditions $speditions, Sites $sites,
	                            IPaymentSpeditionFormFactory $formFactory)
	{
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->paymentsService          = $payments;
		$this->speditionsService        = $speditions;
		$this->sitesService             = $sites;
		$this->formFactory              = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		if ($this->paymentSpeditionsService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.addPaymentSpedition');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.editPaymentSpedition');
		$this['form']->setPaymentSpedition($id);
		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([]);

		$qb = $this->paymentSpeditionsService->getEr()->createQueryBuilder('ps')
			->addSelect('p, s')
			->leftJoin('ps.payment', 'p')
			->leftJoin('ps.spedition', 's')
			->orderBy('ps.site', 'ASC')
			->addOrderBy('s.position', 'ASC')
			->addOrderBy('p.position', 'ASC');
		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name');
		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name');
		$grid->addColumnText('site', 'eshopOrders.paymentSpedition.site', 'site.ident');
		$grid->addColumnStatus('isPublished', 'default.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterSelect('spedition', '', $this->speditionsService->getForSelectOption());
		$grid->addFilterSelect('payment', '', $this->paymentsService->getForSelectOption());
		$grid->addFilterSelect('site', '', [null => ''] + $this->sitesService->getOptionsForSelect());

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype

		return $grid;
	}

	protected function createComponentForm(): PaymentSpeditionForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id'))
			$control->setPaymentSpedition((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[] = function() {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		if ($this->paymentSpeditionsService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

}
