<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Spedition;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\AdminModule\Model\Speditions;

class SpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var Spedition */
	public $spedition;

	/** @var Speditions */
	protected $speditionsService;

	public function __construct(Speditions $speditions)
	{
		$this->speditionsService = $speditions;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.name')->setRequired()->setMaxLength(255);
		$form->addText('ident', 'eshopOrders.paymentForm.itemIdent')->setMaxLength(255);
		$form->addText('text', 'default.text')->setMaxLength(255);
		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addInteger('price', 'eshopOrders.default.priceSimple');
		$form->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
		$form->addInteger('availableFrom', 'eshopOrders.default.availableFrom');
		$form->addInteger('availableTo', 'eshopOrders.default.availableTo');
		$form->addFilesManager('image', 'default.image');
		$form->addText('zboziId', 'eshopOrders.speditionForm.zboziId')->setMaxLength(255);

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			SpeditionForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$spedition    = $this->speditionsService->get($this->id);
				$flashMessage = 'eshopOrders.speditionForm.edited';
			} else {
				$spedition    = new Spedition();
				$flashMessage = 'eshopOrders.speditionForm.added';
			}

			$spedition->setName($values->name);
			$spedition->setIdent($values->ident);
			$spedition->setText($values->text);
			$spedition->isPublished = $values->isPublished;
			$spedition->setPrice((int) $values->price);
			$spedition->setFreeFrom((int) $values->freeFrom);
			$spedition->setAvailableFrom((int) $values->availableFrom);
			$spedition->setAvailableTo((int) $values->availableTo);
			$spedition->image = $values->image;
			$spedition->setZboziId($values->zboziId);

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenter(false) ? $this->template : null,
				$this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $spedition;
			$this->eventDispatcher->dispatch($event, SpeditionForm::class . '::formSuccess');

			$this->em->persist($spedition)->flush();
			$form->addCustomData('speditionId', $spedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setSpedition(int $id): void
	{
		$this->id        = $id;
		$this->spedition = $s = $this->speditionsService->get($id);

		if (!$s)
			throw new InvalidArgumentException();

		$d = [
			'name'          => $s->getName(),
			'ident'         => $s->getIdent(),
			'text'          => $s->getText(),
			'isPublished'   => $s->isPublished,
			'price'         => $s->getPrice(),
			'freeFrom'      => $s->getFreeFrom(),
			'availableFrom' => $s->getAvailableFrom(),
			'availableTo'   => $s->getAvailableTo(),
			'image'         => $s->image,
			'zboziId'       => $s->getZboziId(),
		];

		$this['form']->setDefaults($d);

		$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->spedition), SpeditionForm::class . '::setSpedition');
	}
}
