<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model\Statistics;

use DateTimeInterface;
use EshopOrders\Model\Entities\Order;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Utils\DateTime;

class Statistics
{
	/** @var EntityManagerDecorator */
	protected $em;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
	}

	public function getTodayOrders(): array
	{
		$result = [
			'count' => 0,
			'price' => 0,
		];

		foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
			         ->addSelect('p, s, oi, os, od')
			         ->join('o.payment', 'p')
			         ->join('o.spedition', 's')
			         ->join('o.orderItems', 'oi')
			         ->join('o.orderStatuses', 'os')
			         ->leftJoin('o.orderDiscounts', 'od')
			         ->andWhere('os.created >= :today AND os.status = :status')
			         ->setParameters([
				         'status' => 'created',
				         'today'  => (new DateTime())->setTime(0, 0, 0),
			         ])
			         ->groupBy('o.id')
			         ->getQuery()->getResult() as $order) {
			/** @var Order $order */
			foreach ($order->getOrderStatuses() as $status)
				if ($status->getStatus()->getId() == 'canceled')
					continue 2;

			$result['count']++;
			$result['price'] += $order->getPrice();
		}

		return $result;
	}

	public function getMonthly(?DateTimeInterface $from = null, ?DateTimeInterface $to = null)
	{
		$from = $from ?: (new DateTime())->modify('-1 year');
		$to   = $to ?: (new DateTime());

		$from->modify('first day of this month')->setTime(0, 0, 0);
		$to->modify('last day of this month')->setTime(23, 59, 59);

		$result = [];

		foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
			         ->addSelect('p, s, oi, os, od')
			         ->join('o.payment', 'p')
			         ->join('o.spedition', 's')
			         ->join('o.orderItems', 'oi')
			         ->join('o.orderStatuses', 'os')
			         ->leftJoin('o.orderDiscounts', 'od')
			         ->andWhere('os.created >= :from')->andWhere('os.created <= :to')
			         ->setParameters([
				         'from' => $from,
				         'to'   => $to,
			         ])->getQuery()->getResult() as $order) {
			/** @var Order $order */
			$created = null;

			foreach ($order->getOrderStatuses()->toArray() as $s) {
				if ($s->getStatus()->getId() == 'created')
					$created = $s->getCreated();

				if ($s->getStatus()->getId() == 'canceled')
					continue 2;
			}

			if ($created) {
				$key = $created->format('Y-m');
				if (!isset($result[$key]))
					$result[$key] = $this->monthlyCreateRow($created);

				$result[$key]['orders']++;
				$result[$key]['ordersPrice'] += $order->getPrice();
			}
		}

		krsort($result, SORT_STRING);

		return $result;
	}

	protected function monthlyCreateRow(\DateTime $dateTime)
	{
		return [
			'date'          => $dateTime,
			'orders'        => 0,
			'ordersPrice'   => 0,
			'canceled'      => 0,
			'canceledPrice' => 0,
		];
	}
}
