<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\ControllerPath;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Event\EventDispatcher;
use Core\Model\Countries;
use Core\Model\Entities\Site;
use Core\Model\Sites;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Speditions;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Localization\ITranslator;


/**
 * Class OrdersController
 * @ControllerPath("/orders")
 */
class OrdersController extends BaseController
{
	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var EntityManagerDecorator */
	public $em;

	/** @var Sites */
	public $sitesService;

	/** @var Orders */
	public $ordersService;

	/** @var Payments */
	public $paymentsService;

	/** @var Speditions */
	public $speditionsService;

	/** @var Statuses */
	public $statusesService;

	/** @var Customers */
	public $customersService;

	/** @var Countries */
	public $countriesService;

	/** @var ITranslator */
	public $translator;

	/**
	 * @param EntityManagerDecorator $entityManager
	 * @param ITranslator $translator
	 * @param Countries $countriesService
	 * @param Sites $sitesService
	 * @param Orders $ordersService
	 * @param Customers $customersService
	 * @param Payments $paymentsService
	 * @param Speditions $speditionsService
	 * @param Statuses $statusesService
	 */

	public function __construct(EntityManagerDecorator $entityManager, ITranslator $translator, Countries $countriesService, Sites $sitesService, Orders $ordersService, Customers $customersService, Payments $paymentsService, Speditions $speditionsService, Statuses $statusesService)
	{
		$this->em                = $entityManager;
		$this->sitesService      = $sitesService;
		$this->ordersService     = $ordersService;
		$this->paymentsService   = $paymentsService;
		$this->speditionsService = $speditionsService;
		$this->statusesService   = $statusesService;
		$this->customersService  = $customersService;
		$this->countriesService  = $countriesService;
		$this->translator        = $translator;
	}


	/**
	 * @param ApiRequest $request
	 * @param ApiResponse $response
	 * @return ApiResponse
	 * @Path("/create")
	 * @Method("POST")
	 * @throws \Doctrine\DBAL\ConnectionException
	 */
	public function create(ApiRequest $request, ApiResponse $response)
	{
		$data = $request->getJsonBody();

		try {
			$order = new Order($this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent()));
			$order->setMessage($data['globalNote'] ?? '');
			$order->setAgreedTerms(true);
			$order->setOrderItems($this->convertJSONProductsToOrderItems($data['checkoutProducts'], $order));

			$orderPayment   = new OrderPayment($this->paymentsService->getReferenceByIdent($data['payment']), $order);
			$orderSpedition = new OrderSpedition($this->speditionsService->getReferenceByIdent($data['spedition']), $order);

			$statusCreated      = $this->statusesService->get('finished');
			$orderActionCreated = new OrderStatus($order, $statusCreated);

			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);
			$order->setOrderStatuses([$orderActionCreated]);

			$order->setAddressInvoice($this->createAddress(new OrderAddress(OrderAddress::ADDRESS_INVOICE), $data['customer']['invoice']));
			$order->setAddressDelivery($this->createAddress(new OrderAddress(OrderAddress::ADDRESS_DELIVERY), $data['customer']['delivery']));

			if ($data['customer']['customerId'] && $customer = $this->customersService->get($data['customer']['customerId'])) {
				$order->setCustomer($customer);
			}

			/*if ($data['customer']['customerCreated'] && ($data['customer']['delivery']['email'] || $data['customer']['invoice']['email'])) {
				// Id vkladam jenom formalne - nemel by existovat
				$customer = $this->customersService->getOrCreateCustomer($data['customer']['customerId']);
				$customer->setAddressDelivery($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$customer->setAddressInvoice($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$this->em->persist($customer);
			}*/

			$this->em->getConnection()->beginTransaction();

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			$this->em->persist($orderActionCreated);

			bdump($order);

			$this->em->flush();
			$this->em->getConnection()->commit();

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'success' => $order
			]));

		} catch (\Exception $e) {
			$this->em->getConnection()->rollBack();
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
				'data'  => $data
			]));
		}


	}

	/**
	 * @param $JSONproducts
	 * @param $order
	 * @return array
	 * @throws \Doctrine\ORM\ORMException
	 */
	private function convertJSONProductsToOrderItems($JSONproducts, $order): array
	{
		$result = [];
		foreach ($JSONproducts as $item) {
			$orderItem = new OrderItem($this->em->getReference(Product::class, $item['id']), $order);
			$orderItem->setQuantity($item['count']);
			$orderItem->addOrderItemText($this->translator->getLocale());
			$orderItem->getOrderItemText($this->translator->getLocale())->setName($item['name'])->setVariantName($item['name']);
			$orderItem->setPrice($this->calculatePriceWithSale($item));
			$orderItem->setVatRate($item['vatRate']);
			$orderItem->setCode1($item['code1']);
			$this->em->persist($orderItem);
			$result[] = $orderItem;
		}
		return $result;
	}

	private function calculatePriceWithSale($item)
	{
		$price = $item['basePriceInBaseCurrency'];

		// Pevna sleva na vsechny kusy
		if ($staticSale = ($item['sale']['all']['staticSale'] > 0)) {
			$price -= $staticSale;
		}

		// Procentuelni sleva na vsechny kusy
		if ($percentSale = ($item['sale']['all']['percentSale'] > 0)) {
			$price -= (($price * $item['count']) / 100) * $percentSale;
		}

		// Pevna sleva na kus
		if ($staticSale = ($item['sale']['one']['staticSale'] > 0)) {
			$price -= $staticSale * $item['count'];
		}

		// Procentuelni sleva na kus
		if ($percentSale = ($item['sale']['one']['percentSale'] > 0)) {
			$sale  = (($item['basePriceInBaseCurrency'] / 100) * $percentSale) * $item['count'];
			$price -= $sale;
		}

		return $price;
	}

	/**
	 * @param $address
	 * @param $addressData
	 * @return
	 */
	private function createAddress($address, $addressData)
	{

		$address->setFirstName($addressData['firstName'] ?? '');
		$address->setLastName($addressData['lastName'] ?? '');
		$address->setEmail($addressData['email'] ?? '');
		$address->setPhone($addressData['phone'] ?? '');
		$address->setStreet($addressData['street'] ?? '');
		$address->setCity($addressData['city'] ?? '');
		$address->setPostal($addressData['postal'] ?? '');
		if (!empty($addressData['country'])) {
			$address->setCountry($this->countriesService->getReference($addressData['country']));
		}
		$address->setCompany($addressData['company'] ?? '');
		$address->setIdNumber($addressData['idNumber'] ?? '');
		$address->setVatNumber($addressData['vatNumber'] ?? '');
		$this->em->persist($address);
		bdump($address);
		return $address;
	}

}
