<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\CustomerAddress;
use Users\Model\Entities\User;
use Users\Model\Users;
use Users\Model\Roles;
use EshopOrders\Model\Entities\Customer;

/**
 * class Customers
 * @package EshopOrders\FrontModule\Model
 *
 * @method Customer|object|null getReference($id)
 * @method Customer[]|null getAll()
 * @method Customer|null get($id)
 */
class Customers extends BaseEntityService
{
	const CUSTOMER = 'customer';

	protected $entityClass = Customer::class;

	/** @var Roles */
	protected $rolesService;

	/** @var Users */
	private $usersService;

	public function __construct(Users $users, Roles $roles)
	{
		$this->usersService = $users;
		$this->rolesService = $roles;
	}

	/**
	 * @param User $user
	 *
	 * @return Customer|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUser(User $user): ?Customer
	{
		return $this->getEr()->createQueryBuilder('c', 'c.id')
			->andWhere('c.user = :user')->setParameter('user', $user)
			->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param int $userId
	 *
	 * @return Customer|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUserId(int $userId): ?Customer
	{
		return $this->getByUser($this->usersService->getReference($userId)) ?? null;
	}

	public function getOrCreateCustomer($userId, $email = null, $firstName = '', $lastName = '', $phone = '')
	{
		//user
		$user = $this->usersService->getOrCreateUser($userId, $email, $firstName, $lastName);

		//ma user prava "customer"? Pokud ne, tak pridat prava
		if (!$user->isInRoleByIdent(self::CUSTOMER)) {
			$role = $this->rolesService->getByIdent(self::CUSTOMER);
			if ($role) {
				$user->addRole($role);
				$this->em->persist($user)->flush();
			}
		}

		//customer
		$customer = $this->getByUser($user);
		if (!$customer) {
			$customer        = new Customer($user, $phone);
			$addressDelivery = new CustomerAddress($customer);
			$customer->setAddressDelivery($addressDelivery);
			$this->em->persist($customer);
			$this->em->persist($addressDelivery);
			$this->em->flush();
		}

		return $customer;
	}
}
