<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

class Cart
{

	/** @var int */
	public $id;

	/** @var string */
	public $ident;

	/** @var CartItem[] */
	public $cartItems;

	/** @var Spedition */
	public $spedition;

	/** @var Payment */
	public $payment;

	/** @var Discount[] */
	public $discounts = [];

	/** @var Gift[] */
	public $gifts = [];

	/** @var array */
	protected $cCalculatedDiscounts = [];

	public function setId(int $id)
	{
		$this->id = $id;

		return $this;
	}

	public function setIdent($ident)
	{
		$this->ident = $ident;

		return $this;
	}

	public function setCartItems($cartItems)
	{
		$this->cartItems = $cartItems;

		return $this;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function getIdent()
	{
		return $this->ident;
	}

	/** @return CartItem[] */
	public function getCartItems(): array
	{
		return $this->cartItems;
	}

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function addGift(Gift $gift): self
	{
		$this->gifts[$gift->id] = $gift;

		return $this;
	}

	public function getCartItemsPrice($withVat = true): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item)
			$priceTotal += $withVat ? $item->getTotalPrice() : $item->getTotalPriceWithoutVat();

		return $priceTotal;
	}

	public function getPriceTotal(): float
	{
		$priceTotal = $this->getCartItemsPrice(true);
		$priceTotal += $this->calculateDiscounts(true);

		if ($this->spedition)
			$priceTotal += $this->spedition->getPriceActual($this);
		if ($this->payment)
			$priceTotal += $this->payment->getPriceActual($this);

		return $priceTotal;
	}

	public function getPriceTotalWithoutVat(): float
	{
		$priceTotal = $this->getCartItemsPrice(false);
		$priceTotal += $this->calculateDiscounts(false);

		if ($this->spedition)
			$priceTotal += $this->spedition->getPriceActual($this);
		if ($this->payment)
			$priceTotal += $this->payment->getPriceActual($this);

		return $priceTotal;
	}

	public function getPriceTotalWithoutSpedition($withVat = true): float
	{
		return $withVat
			? $this->getCartItemsPrice(true) + $this->calculateDiscounts(true)
			: $this->getCartItemsPrice(false) + $this->calculateDiscounts(false);
	}

	public function getItemsCount(): int
	{
		$count = 0;
		foreach ($this->cartItems as $item) {
			$count += $item->getQuantity();
		}

		return $count;
	}

	public function addDiscount(string $key, Discount $discount): self
	{
		$this->discounts[$key] = $discount;
		$discount->calculateDiscount($this->getCartItemsPrice());

		return $this;
	}

	public function removeDiscount(string $key): self
	{
		unset($this->discounts[$key]);

		return $this;
	}

	public function getDiscount(string $key): ?Discount { return $this->discounts[$key] ?? null; }

	/** @return Discount[] */
	public function getDiscounts(): iterable
	{
		// Přepočítá slevu, může tam být uložena sleva bez DPH
		$this->calculateDiscounts();

		$discounts = [];
		foreach ($this->discounts as $k => $discount) {
			if ($discount->discount == 0)
				continue;

			$discounts[$k] = $discount;
		}

		return $discounts;
	}

	public function setSpedition(Spedition $spedition): self
	{
		$spedition->setCart($this);
		$this->spedition = $spedition;

		return $this;
	}

	public function setPayment(Payment $payment): self
	{
		$payment->setCart($this);
		$this->payment = $payment;

		return $this;
	}

	public function calculateDiscounts($withVat = true): float
	{
		$productsPrice = 0;

		foreach ($this->getProductsForDiscount() as $item)
			$productsPrice += $withVat ? $item->getTotalPrice() : $item->getTotalPriceWithoutVat();

		$k = ceil($productsPrice * 1000);
		if (!isset($this->cCalculatedDiscounts[$k])) {
			$discountAmount = 0;
			foreach ($this->discounts as $discount)
				$discountAmount += $discount->calculateDiscount($productsPrice + $discountAmount);

			$this->cCalculatedDiscounts[$k] = $discountAmount;
		}

		return $this->cCalculatedDiscounts[$k];
	}

	public function calculateDiscount(Discount $discount, $withVat = true, bool $useBaseCurrency = false): float
	{
		$productsPrice = 0;

		foreach ($this->getProductsForDiscount() as $item) {
			$productsPrice += $withVat ? $item->getTotalPrice($useBaseCurrency) : $item->getTotalPriceWithoutVat($useBaseCurrency);
		}

		return $discount->calculateDiscount($productsPrice, $useBaseCurrency);
	}

	/** @return CartItem[] */
	protected function getProductsForDiscount(): array
	{
		$items = [];

		foreach ($this->cartItems as $k => $item) {
			if ($item->discountDisabled)
				continue;

			$items[$k] = $item;
		}

		return $items;
	}
}


