<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\FrontModule\Model\Dao;

/**
 * Class Payments
 * @package EshopOrders\FrontModule\Model
 */
class Payments extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Payment::class;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var Dao\Payment[] */
	protected $cPublished;

	/**
	 * @param int $id
	 *
	 * @return Dao\Payment|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$paymentRaw = $this->loadRawPayment($id);
		if (!$paymentRaw) {
			return null;
		}

		return $this->fillDao([$paymentRaw])[$id];
	}

	/** Pozor - vraci dopravy nezavisle na hodnote kosiku, takze i ty ktere nemusi byt platne
	 * @return Dao\Payment[] vsechny aktivni dopravy
	 */
	public function getAllPublished()
	{
		if ($this->cPublished === null) {
			$paymentsRaw = $this->getEr()->createQueryBuilder('p', 'p.id')
				->andWhere('p.isPublished = :isPublished')
				->setParameters([
					'isPublished' => 1,
				])
				->orderBy('p.position')
				->getQuery()->getResult();

			$this->cPublished = $this->fillDao($paymentsRaw);
		}

		return $this->cPublished;
	}

	/**
	 * @param Dao\Payment[] $payments
	 * @param float         $cartValue
	 *
	 * @return Dao\Payment[]
	 */
	public function filterByCartValue(array $payments, float $cartValue): array
	{
		foreach ($payments as $k => $payment) {
			if ($payment->getAvailableFrom() > $cartValue || $payment->getAvailableTo() < $cartValue)
				unset($payments[$k]);
		}

		return $payments;
	}

	public function getFirstFreePayment()
	{
		$paymentQuery = $this->getEr()->createQueryBuilder('p', 'p.id')->andWhere('p.freeFrom > 0')->setMaxResults(1)->orderBy('p.freeFrom', 'ASC');
		$paymentRaw   = $paymentQuery->getQuery()->getOneOrNullResult();

		return $paymentRaw ? current($this->fillDao([$paymentRaw])) : null;
	}

	public function getRawReference($id)
	{
		return $this->getEr()->getReference($id);
	}

	/**
	 * @param $id
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	private function loadRawPayment($id)
	{
		$paymentRawQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentRawQuery->andWhere('p.id = :id')->setParameter('id', $id);

		return $paymentRawQuery->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param Payment[] $paymentsRaw
	 *
	 * @return Dao\Payment[]
	 */
	public function fillDao($paymentsRaw)
	{
		$payments = [];
		foreach ($paymentsRaw as $v) {
			$payment = new Dao\Payment();

			$payment
				->setId($v->getId())
				->setIdent($v->getIdent())
				->setName($v->getName())
				->setText($v->getText())
				->setPublished($v->isPublished)
				->setPosition($v->getPosition())
				->setPrice((float) $v->getPrice())
				->setFreeFrom((float) $v->getFreeFrom())
				->setAvailableFrom($v->getAvailableFrom())
				->setAvailableTo($v->getAvailableTo())
				->setImage($v->image);

			$payment->priceInBaseCurrency    = $v->getPrice();
			$payment->freeFromInBaseCurrency = $v->getFreeFrom();

			$payments[$payment->getId()] = $payment;
		}

		$this->eventDispatcher->dispatch(new PaymentsSpeditionsEvent($payments, null), Payments::class . '::afterFillDao');

		return $payments;
	}
}

