<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\FrontModule\Model\Dao;
use Doctrine\ORM\Query\Expr\Join;
/**
 * Class Speditions
 * @package EshopOrders\FrontModule\Model
 */
class Speditions extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Spedition::class;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var Sites @inject */
	public $sitesService;

	/** @var Dao\Spedition[] */
	protected $cPublished;

	/**
	 * @param int $id
	 *
	 * @return Dao\Spedition|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$speditionRaw = $this->loadRawSpedition($id);
		if (!$speditionRaw) {
			return null;
		}
		$spedition = $this->fillDao([$speditionRaw])[$id];

		return $spedition;
	}

	/** Pozor - vraci dopravy nezavisle na hodnote kosiku, takze i ty ktere nemusi byt platne
	 * @return Dao\Spedition[] vsechny aktivni dopravy
	 */
	public function getAllPublished()
	{
		if ($this->cPublished === null) {
			$speditionsRaw = $this->getEr()->createQueryBuilder('s', 's.id')
				->andWhere('s.isPublished = :isPublished')
				->setParameters([
					'isPublished' => 1,
				])
				->orderBy('s.position')
				->getQuery()->getResult();

			$this->cPublished = $this->fillDao($speditionsRaw);
		}

		return $this->cPublished;
	}

	/**
	 * @param Dao\Spedition[] $speditions
	 * @param float           $cartValue
	 *
	 * @return Dao\Spedition[]
	 */
	public function filterByCartValue(array $speditions, float $cartValue): array
	{
		foreach ($speditions as $k => $spedition) {
			if ($spedition->getAvailableFrom() > $cartValue || $spedition->getAvailableTo() < $cartValue)
				unset($speditions[$k]);
		}

		return $speditions;
	}

	public function getFirstFreeSpedition()
	{
		$speditionQuery = $this->getEr()->createQueryBuilder('s', 's.id')
			->andWhere('s.freeFrom > 0')
			->setMaxResults(1)
			->innerJoin(PaymentSpedition::class, 'ps', Join::WITH, 'ps.spedition = s.id AND ps.site = :site')
			->setParameters([
				'site' => $this->sitesService->getCurrentSite()->getIdent(),
			])
			->orderBy('s.freeFrom', 'ASC');
		$speditionRaw   = $speditionQuery->getQuery()->getOneOrNullResult();

		return $speditionRaw ? current($this->fillDao([$speditionRaw])) : null;
	}

	public function getRawReference($id)
	{
		return $this->getEr()->getReference($id);
	}

	/**
	 * @param $id
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	private function loadRawSpedition($id)
	{
		$speditionRawQuery = $this->getEr()->createQueryBuilder('s', 's.id');
		$speditionRawQuery->andWhere('s.id = :id')->setParameter('id', $id);

		return $speditionRawQuery->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param Spedition[] $speditionsRaw
	 *
	 * @return Dao\Spedition[]
	 */
	public function fillDao($speditionsRaw)
	{
		$speditions = [];
		foreach ($speditionsRaw as $spedRaw) {
			$spedition = new Dao\Spedition();

			$spedition
				->setId($spedRaw->getId())
				->setIdent($spedRaw->getIdent())
				->setName($spedRaw->getName())
				->setText($spedRaw->getText())
				->setPublished($spedRaw->isPublished)
				->setPosition($spedRaw->getPosition())
				->setPrice((float) $spedRaw->getPrice())
				->setFreeFrom((float) $spedRaw->getFreeFrom())
				->setAvailableFrom($spedRaw->getAvailableFrom())
				->setAvailableTo($spedRaw->getAvailableTo())
				->setImage($spedRaw->image);

			$spedition->priceInBaseCurrency    = $spedRaw->getPrice();
			$spedition->freeFromInBaseCurrency = $spedRaw->getFreeFrom();

			$speditions[$spedition->getId()] = $spedition;
		}

		$this->eventDispatcher->dispatch(new PaymentsSpeditionsEvent(null, $speditions), Speditions::class . '::afterFillDao');

		return $speditions;
	}
}

