<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Nette\Application\LinkGenerator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\PresenterTemplateEvent;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\Entities\CartItem;
use Nette\Localization\ITranslator;

class CartSubscriber implements EventSubscriberInterface
{
	/** @var Carts */
	protected $cartsService;

	/** @var ProductsFacade */
	protected $productsFacade;

	/** @var Tags */
	protected $tagsService;

	/** @var ITranslator */
	protected $translator;

	/** @var LinkGenerator */
	protected $linkGenerator;

	/** @var array */
	protected $cartChanges = [];

	public function __construct(Carts $carts, ProductsFacade $productsFacade, Tags $tags, ITranslator $translator, LinkGenerator $linkGenerator)
	{
		$this->cartsService   = $carts;
		$this->productsFacade = $productsFacade;
		$this->tagsService    = $tags;
		$this->translator     = $translator;
		$this->linkGenerator  = $linkGenerator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'core.front.beforeRender'      => 'beforeRender',
			'eshopOrders.cartAddItem'      => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'   => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'   => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems' => ['onFillDaoItems', 100],
		];
	}

	public function beforeRender(PresenterTemplateEvent $event): void
	{
		if (!empty($this->cartChanges)) {
			$message = null;
			$event->presenter->redrawControl('orderCartDetail');
			$event->presenter->redrawControl('cartPreview');
			$event->presenter->redrawControl('cartPreviewMobile');

			if (isset($this->cartChanges['added']))
				$message = 'itemAdded';
			if (isset($this->cartChanges['updated']))
				$message = 'itemUpdated';
			if (isset($this->cartChanges['removed']))
				$message = 'itemRemoved';
			if ($message)
				$event->presenter->flashMessageSuccess('eshopOrdersFront.cart.' . $message);

			$event->presenter->redrawControl('flashes');
		}
	}

	public function onAddItem(AddedCartItemEvent $event): void
	{
		$item = $event->item;

		$this->cartChanges['added'][] = $item;
		$this->cartsService->addItem($item);
	}

	public function onUpdateItem(UpdatedCartItemEvent $event): void
	{
		$this->cartChanges['updated'][] = $event->itemId;
		$this->cartsService->updateItemQuantity($event->itemId, $event->quantity);
	}

	public function onRemoveItem(RemovedCartItemEvent $event): void
	{
		$this->cartChanges['removed'][] = $event->itemId;
		$this->cartsService->removeItem($event->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems)
			return;

		$productIds = array_map(function(CartItem $ci) { return $ci->getProductId(); }, $cartItems);
		$products   = $this->productsFacade->getProducts($productIds);

		$items = [];
		foreach ($cartItems as $ci) {
			$product = $products[$ci->getProductId()] ?? null;

			if (!$products)
				continue;

			$cartItem = new Dao\CartItem((string) $product->getName(), $product->getId(), $product->getPrice());

			$cartItem
				->setId($ci->getId())
				->setProductId($ci->getProductId())
				->setProduct($product)
				->setVariantId($ci->getVariantId())
				->setProductVariant(null)
				->setQuantity($ci->getQuantity())
				->setLink($product->link)
				->setVatRate((int) $product->getVatRate());

			$cartItem->priceInBaseCurrency = $product->priceInBaseCurrency;
			$cartItem->discountDisabled    = $product->discountDisabled;

			if ($product->getGallery() && $product->getGallery()->getCover())
				$cartItem->setImage($product->getGallery()->getCover());

			$tagFree = $product->getTag('freeDelivery');
			if ($tagFree && !$tagFree->isAuto)
				$cartItem->freeDelivery = true;

			$items[$ci->getId()] = $cartItem;
		}

		$this->cartsService->cDaoItems = $items;
	}
}
