<?php declare(strict_types=1);

namespace EshopOrders\Model\Entities\Invoice;

use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__invoice_product")
 * @ORM\Entity
 */
class Product
{
	use TId;

	/**
	 * @var string|null
	 * @ORM\Column(type="string", nullable = true)
	 */
	public $name;

	/**
	 * Parametr podstatny jen pri vytvareni faktury, jinak nepouzivat
	 * @var integer|null
	 * @ORM\Column(type="integer")
	 */
	public $productId;

	/**
	 * @var integer
	 * @ORM\Column(type="integer", options={"default": 0})
	 */
	public $quantity = 0;

	/**
	 * @var float
	 * @ORM\Column(type="decimal", precision=10, scale=2)
	 */
	public $price = 0;

	/**
	 * @var int
	 * @ORM\Column(type="smallint")
	 */
	public $vatRate;

	/**
	 * @var string
	 * @ORM\Column(type="text", nullable=true)
	 */
	public $imageBase64;

	/**
	 * @var InvoiceData
	 *
	 * @ORM\ManyToOne(targetEntity="InvoiceData")
	 * @ORM\JoinColumn(name="invoice_data_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $invoiceData;

	/**
	 * Product constructor.
	 * @param InvoiceData $invoiceData
	 */
	public function __construct(InvoiceData $invoiceData)
	{
		$this->invoiceData = $invoiceData;
	}

	/**
	 * @return string
	 */
	public function getImageBase64(): string
	{
		return str_replace('png', 'jpeg', $this->imageBase64); // png mi nechce mpdf zobrazit
	}

	/**
	 * @return float
	 */
	public function getPrice(): float
	{
		return (float) $this->price;
	}

	/**
	 * @return float
	 */
	public function getPriceTotal(): float
	{
		return (float)  $this->price * $this->quantity;
	}

	/**
	 * @return float
	 */
	public function getTotalPriceWithoutVat(): float
	{
		return round($this->getPriceWithoutVat() * $this->quantity, 2);
	}

	/**
	 * @return float
	 */
	public function getPriceWithoutVat(): float
	{
		return round($this->price / (1 + ($this->vatRate / 100)), 2);
	}


}
