<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Templating\Filters\Price as PriceFilter;
use Currency\DI\CurrencyExtension;
use Currency\Model\Config as CurrencyConfig;
use Core\Model\Entities\Site;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"OrderListener"})
 */
class Order
{
	use TId;

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string", length=255, nullable = false)
	 */
	protected $ident;

	/**
	 * @var Customer
	 * @ORM\ManyToOne(targetEntity="Customer")
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="SET NULL", nullable = true)
	 */
	protected $customer;

	/**
	 * @var string
	 * @ORM\Column(name="message", type="text", nullable = false)
	 */
	protected $message;

	/**
	 * @var OrderPayment
	 * @ORM\OneToOne(targetEntity="OrderPayment")
	 * @ORM\JoinColumn(name="payment", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $payment;

	/**
	 * @var OrderSpedition
	 * @ORM\OneToOne(targetEntity="OrderSpedition")
	 * @ORM\JoinColumn(name="spedition", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $spedition;

	/**
	 * @var OrderItem[]
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="order", indexBy="id")
	 */
	protected $orderItems;

	/**
	 * @var OrderDiscount[]
	 * @ORM\OneToMany(targetEntity="OrderDiscount", mappedBy="order", indexBy="id")
	 */
	protected $orderDiscounts;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_delivery_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressDelivery;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressInvoice;

	/**
	 * @var OrderStatus[]
	 * @ORM\OneToMany(targetEntity="OrderStatus", mappedBy="order")
	 */
	protected $orderStatuses;

	/**
	 * @var boolean
	 * @ORM\Column(name="agreed_terms", type="boolean", nullable=true)
	 */
	protected $agreedTerms;

	/**
	 * @var OrderFlag[]
	 * @ORM\OneToMany(targetEntity="OrderFlag", mappedBy="order", indexBy="type")
	 */
	protected $orderFlags;

	/**
	 * @var Invoice|null
	 * @ORM\OneToOne(targetEntity="Invoice", inversedBy="order")
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $invoice;

	/**
	 * @var Site
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Site")
	 * @ORM\JoinColumn(name="site_id", referencedColumnName="ident", onDelete="SET NULL", nullable=true)
	 */
	public $site;

	/**
	 * @var OrderCurrency|null
	 * @ORM\OneToOne(targetEntity="OrderCurrency", mappedBy="order", cascade={"persist"})
	 */
	public $currency;

	/**
	 * @var OrderGift[]
	 * @ORM\OneToMany(targetEntity="OrderGift", mappedBy="order")
	 */
	protected $gifts;

	/** @var PriceFilter */
	public $priceFilter;

	public function __construct(Site $site)
	{
		$this->ident           = $this->genUuid();
		$this->site            = $site;
		$this->orderItems      = new ArrayCollection();
		$this->orderDiscounts  = new ArrayCollection();
		$this->orderFlags      = new ArrayCollection();
		$this->gifts           = new ArrayCollection();
		$this->addressDelivery = null;
		$this->addressInvoice  = null;
	}

	public function getIdent()
	{
		return $this->ident;
	}

	/**
	 * @return Customer
	 */
	public function getCustomer()
	{
		return $this->customer;
	}

	public function getMessage()
	{
		return $this->message;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency)
			+ $this->spedition->getPrice($useCurrency)
			+ $this->payment->getPrice($useCurrency);

		foreach ($this->orderDiscounts as $discount) {
			$priceTotal += $discount->calculateDiscount($priceTotal, $useCurrency);
		}

		return $priceTotal;
	}

	public function getPriceItems(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->orderItems as $item) {
			$priceTotal += $item->getPriceTotal($useCurrency);
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount(bool $useCurrency = false): float
	{
		// TODO currency
		$priceTotal = $this->getPriceItems();
		if ($this->orderDiscounts) {
			foreach ($this->orderDiscounts as $discount) {
				$priceTotal += $discount->getPrice();
			}
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat()
	{
		$priceTotal = $this->calculateDiscounts($this->getItemsPriceWithoutVat());

		if ($this->spedition)
			$priceTotal += $this->spedition->getPrice();
		if ($this->payment)
			$priceTotal += $this->payment->getPrice();

		return $priceTotal;
	}

	public function getItemsPriceWithoutVat(): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPaySpedPrice()
	{
		$priceTotal = 0;
		$priceTotal += $this->spedition->getPrice();
		$priceTotal += $this->payment->getPrice();

		return $priceTotal;
	}

	/**
	 * @return OrderPayment
	 */
	public function getPayment()
	{
		return $this->payment;
	}

	/**
	 * @return OrderSpedition
	 */
	public function getSpedition()
	{
		return $this->spedition;
	}

	/**
	 * @return OrderItem[]
	 */
	public function getOrderItems()
	{
		return $this->orderItems;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressDelivery()
	{
		return $this->addressDelivery ?: $this->getAddressInvoice();
	}

	public function getAddressDeliveryRaw()
	{
		return $this->addressDelivery;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressInvoice()
	{
		return $this->addressInvoice ?: ($this->addressDelivery ?: null);
	}

	/**
	 * @param Customer $customer
	 *
	 * @return Order
	 */
	public function setCustomer(Customer $customer): Order
	{
		$this->customer = $customer;

		return $this;
	}

	public function setMessage($message)
	{
		$this->message = $message;

		return $this;
	}

	/**
	 * @param OrderPayment $payment
	 *
	 * @return Order
	 */
	public function setPayment($payment)
	{
		$this->payment = $payment;

		return $this;
	}

	/**
	 * @param OrderSpedition $spedition
	 *
	 * @return Order
	 */
	public function setSpedition($spedition)
	{
		$this->spedition = $spedition;

		return $this;
	}

	public function setOrderItems(array $orderItems)
	{
		$this->orderItems = new ArrayCollection($orderItems);

		return $this;
	}

	public function setAddressDelivery(OrderAddress $addressDelivery)
	{
		$this->addressDelivery = $addressDelivery;

		return $this;
	}

	public function setAddressInvoice(OrderAddress $addressInvoice)
	{
		$this->addressInvoice = $addressInvoice;

		return $this;
	}

	/*******
	 * === OrderDiscounts
	 */

	public function getOrderDiscounts()
	{
		return $this->orderDiscounts;
	}

	public function setOrderDiscounts($orderDiscounts): Order
	{
		$this->orderDiscounts = $orderDiscounts;

		return $this;
	}

	public function addOrderDiscount($orderDiscount): Order
	{
		$this->orderDiscounts->add($orderDiscount);

		return $this;
	}

	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->getOrderDiscounts() as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	/*******
	 * === AgreedTerms
	 */

	public function getAgreedTerms()
	{
		return $this->agreedTerms;
	}

	public function setAgreedTerms($agreedTerms): Order
	{
		$this->agreedTerms = $agreedTerms;

		return $this;
	}

	/*******
	 * === OrderFlags
	 */

	public function getOrderFlags()
	{
		return $this->orderFlags;
	}

	public function setOrderFlags($orderFlags): Order
	{
		if (is_array($orderFlags))
			$orderFlags = new ArrayCollection($orderFlags);
		$this->orderFlags = $orderFlags;

		return $this;
	}

	public function addFlag(OrderFlag $flag)
	{
		if (!$this->orderFlags->containsKey($flag->getType())) {
			$this->orderFlags->set($flag->getType(), $flag);
		}
	}

	public function hasFlag($type)
	{
		return ($this->orderFlags->containsKey($type) && $this->orderFlags[$type] == true);
	}

	/*******
	 * === OrderStatuses
	 */

	public function getOrderStatuses()
	{
		return $this->orderStatuses;
	}

	public function getNewestOrderStatus()
	{
		$newestStatus = $this->orderStatuses->last();

		return $newestStatus;
	}

	public function setOrderStatuses($orderStatuses): Order
	{
		$this->orderStatuses = $orderStatuses;

		return $this;
	}

	public function getCreatedTime()
	{
		foreach ($this->orderStatuses as $orderStatus) {
			if ($orderStatus->getStatus()->getId() == 'created') {
				$createdStatus = $orderStatus;
				break;
			}
		}
		if ($createdStatus) {
			return $createdStatus->getCreated();
		}

		return false;
	}

	public function getInvoice(): ?Invoice
	{
		return $this->invoice;
	}

	public function setInvoice(Invoice $invoice)
	{
		$this->invoice = $invoice;
	}

	public function getCurrencyCode(): string
	{
		if (class_exists(CurrencyExtension::class) && $this->currency)
			return $this->currency->code;

		return $this->getDefaultCurrency();
	}

	public function getDefaultCurrency() { return class_exists(CurrencyConfig::class) ? CurrencyConfig::load('default') : 'CZK'; }

	public function calculateCurrencyPrice(float $price): float
	{
		return $this->currency
			? round($price * (1 / $this->currency->rate), 2)
			: $price;
	}

	public function renderPrice(float $price, bool $useCurrency = false, string $method = 'format')
	{
		return $this->priceFilter->$method($price, $useCurrency ? $this->getCurrencyCode() : $this->getDefaultCurrency());
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection|OrderGift[] */
	public function getGifts() { return $this->gifts; }

	public function addGift(OrderGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid()
	{
		return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff),

			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,

			// 48 bits for "node"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
		);
	}
}
