<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__order_item")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class OrderItem
{
	use TId;
	use TTranslateListener;

	/**
	 * @var Product
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Product")
	 * @ORM\JoinColumn(name="product_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $product;

	/**
	 * @var int
	 * @ORM\Column(name="variant_id", type="integer", nullable=true)
	 */
	protected $variantId;

	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	protected $code1;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="smallint")
	 */
	protected $quantity;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @var int
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public $vatRate;

	/**
	 * @var Order
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderItems")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $order;

	/**
	 * @var OrderItemTexts[]
	 * @ORM\OneToMany(targetEntity="OrderItemTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 * TODO moznost optimalizace - nekaskadovat
	 */
	protected $orderItemTexts;

	/**
	 * @var OrderItemGift[]
	 * @ORM\OneToMany(targetEntity="OrderItemGift", mappedBy="orderItem")
	 */
	protected $gifts;

	/**
	 * @var OrderItemSale[]
	 * @ORM\OneToMany(targetEntity="OrderItemSale", mappedBy="orderItem")
	 */
	public $sales;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	public function __construct(Product $product, Order $order)
	{
		$this->variantId      = null;
		$this->product        = $product;
		$this->order          = $order;
		$this->quantity       = 1;
		$this->orderItemTexts = new ArrayCollection();
		$this->moreData       = [];
		$this->gifts          = new ArrayCollection();
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity($quantity)
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity()
	{
		return $this->quantity;
	}

	/*******
	 * === Product
	 */

	public function getProduct()
	{
		return $this->product;
	}

	public function getProductId()
	{
		return $this->product ? $this->product->getId() : null;
	}

	public function setProductId($productId)
	{
		$this->productId = $productId;

		return $this;
	}

	/*******
	 * === Variant
	 */

	public function getVariantId()
	{
		return $this->variantId;
	}

	public function setVariantId($variantId)
	{
		$this->variantId = $variantId;

		return $this;
	}

	/*******
	 * === Code1
	 */

	public function getCode1()
	{
		return $this->code1;
	}

	public function setCode1($code1): OrderItem
	{
		$this->code1 = $code1;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order)
	{
		$this->order = $order;

		return $this;
	}

	/*******
	 * === Price
	 */

	public function getPrice(bool $useCurrency = false): float
	{
		return $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice((float)$this->price)
			: (float)$this->price;
	}

	public function getPriceWithoutSales(bool $useCurrency = false): float
	{
		return (float)($this->getPrice($useCurrency) * $this->quantity);
	}

	public function getPriceTotal(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency) * $this->quantity;
		foreach ($this->sales as $s) {
			$price -= $s->getSaleValue($useCurrency);
		}
		return (float)$price;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		return round($this->getPrice($useCurrency) / (1 + ($this->getVatRate() / 100)), 2);
	}

	public function getTotalPriceWithoutVat(bool $useCurrency = false): float
	{
		return round($this->getPriceWithoutVat($useCurrency) * $this->getQuantity(), 2);
	}

	public function setPrice(float $price)
	{
		// Asi bug, float neprojde
		$price = str_replace(',', '.', $price);

		$this->price = $price;

		return $this;
	}

	/*******
	 * === VatRate
	 */

	public function getVatRate()
	{
		return $this->vatRate;
	}

	public function setVatRate($vatRate): OrderItem
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/*******
	 * === OrderItemText
	 */

	public function addOrderItemText($lang)
	{
		$this->orderItemTexts->set($lang, new OrderItemTexts($this, $lang));
	}

	public function setOrderItemText(OrderItemTexts $orderItemTexts)
	{
		$this->orderItemTexts->set($orderItemTexts->getLang(), $orderItemTexts);
	}

	/**
	 * @param string|null $lang
	 *
	 * @return OrderItemTexts|null
	 */
	public function getOrderItemText(?string $lang = null): ?OrderItemTexts
	{
		return $this->orderItemTexts->get($lang ?: $this->locale) ?? $this->orderItemTexts->first();
	}


	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key)
	{
		return $this->moreData[$key] ?? null;
	}

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection|OrderItemGift[] */
	public function getGifts()
	{
		return $this->gifts;
	}

	public function addGift(OrderItemGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}
}

