<?php declare(strict_types=1);

namespace EshopOrders\Model;

use Core\Model\Entities\Site;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\Entities\SiteInvoiceConfig;
use Exception;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class InvoiceConfigRepository extends BaseEntityService
{
	/** @var string */
	protected $entityClass = InvoiceConfig::class;

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('ic');
	}

	/**
	 * @param string $siteIdent
	 * @return DateTime|null
	 * @throws Exception
	 */
	public function getDueDate(string $siteIdent): ?DateTime
	{
		$now = new DateTime;
		$defaultConfig = $this->getConfigBySite($siteIdent);

		if (!$defaultConfig) {
			return null;
		}

		return $now->modify(sprintf('+ %s days', $defaultConfig->maturity));
	}

	/**
	 * @param string $siteIdent
	 * @return string|null
	 * @throws NonUniqueResultException
	 */
	public function generateIdent(string $siteIdent): ?string
	{
		$defaultConfig = $this->getConfigBySite($siteIdent);
		if (!$defaultConfig) {
			return null;
		}
		$numericalSeries = $defaultConfig->numericalSeries;
		$containsPrefixYearWildcards = $numericalSeries->containsPrefixYearWildcards();
		$containsPrefixMonthWildcards = $numericalSeries->containsPrefixMonthWildcards();
		$startNumber = $numericalSeries->startNumber;

		if ($containsPrefixMonthWildcards || $containsPrefixYearWildcards) {

			/** @var Invoice $lastInvoice */
			$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
			$qb->join('i.site', 's')
				->where($qb->expr()->in('s.ident', array_map(static function (SiteInvoiceConfig $config) {
					return $config->site->getIdent();
				}, $defaultConfig->siteInvoiceConfigs->toArray())));

			$lastInvoice = $this->em->getRepository(Invoice::class)->findOneBy([], ['ident' => 'desc']);

			if ($lastInvoice !== null) {
				$created = $lastInvoice->order->getCreatedTime();
				$now = new DateTime;

				// prefix contain month/year wildcards and current month/year is bigger then on the invoice
				if (($containsPrefixMonthWildcards && (((int) $now->format('n')) > ((int) $created->format('n')))) ||
					($containsPrefixYearWildcards && (((int) $now->format('Y')) > ((int) $created->format('Y')))))
				{
					$this->resetInvoiceCounter();
					$startNumber = 1;
				}
			}

		}

		$strMaxOrder = Strings::padRight('9', $numericalSeries->digitsCount, '9');

		if (((int) $strMaxOrder) === $startNumber) {
			$lastInvoice = $this->getLastInvoice();

			if (Strings::endsWith($lastInvoice->ident, $strMaxOrder)) {
				return null;
			}
		}

		$prefix = $numericalSeries->getRealPrefix();
		$order = Strings::padLeft($startNumber, $numericalSeries->digitsCount, '0');

		return sprintf('%s%s', $prefix, $order);
	}

	/**
	 * @param string $siteIdent
	 * @throws Exception
	 */
	private function resetInvoiceCounter(string $siteIdent): void
	{
		$defaultConfig = $this->getConfigBySite($siteIdent);

		if (!$defaultConfig) {
			return;
		}

		$numericalSeries = $defaultConfig->numericalSeries;
		$numericalSeries->startNumber = 1;

		$this->em->persist($numericalSeries);
		$this->em->flush();
	}

	/**
	 * @return SiteInvoiceConfig[]
	 */
	public function getUsed(): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sic')
			->from(SiteInvoiceConfig::class, 'sic')
			->join('sic.site', 's');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @return bool
	 */
	public function haveAllSitesInvoiceSettingsCreated(): bool
	{
		return $this->em->getRepository(Site::class)->countBy() === count($this->getUsed());
	}

	/**
	 * @param string $siteIdent
	 * @return InvoiceConfig|null
	 * @throws NonUniqueResultException
	 */
	public function getConfigBySite(string $siteIdent): ?InvoiceConfig
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('ic')
			->from(InvoiceConfig::class, 'ic')
			->join('ic.siteInvoiceConfigs', 'sic')
			->join('sic.site', 's')
			->where('s.ident = :ident')
			->setParameter('ident', $siteIdent);

		return $qb->getQuery()->getOneOrNullResult();
	}
	
}
