<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImageHelper;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\AdminModule\Model\Event\InvoiceRegenerateEvent;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use Exception;
use Doctrine\ORM\Query\Expr\Join;
use Core\Model\Event\EventDispatcher;
use Nette\Localization\ITranslator;
use Nette\Utils\Image;
use Tracy\Debugger;

/**
 * Class Invoices
 * @package EshopOrders\Model
 *
 * @method Invoice|null get($id)
 */
class Invoices extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Invoice::class;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var Products */
	protected $products;

	/** @var ITranslator */
	protected $translator;

	/** @var Sellers */
	protected $sellers;

	/**
	 * Invoices constructor.
	 *
	 * @param EventDispatcher $eventDispatcher
	 * @param Products        $products
	 * @param ITranslator     $translator
	 * @param Sellers         $sellers
	 */
	public function __construct(EventDispatcher $eventDispatcher, Products $products, ITranslator $translator, Sellers $sellers)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->products        = $products;
		$this->translator      = $translator;
		$this->sellers         = $sellers;
	}

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('i')->orderBy('i.ident', 'desc');
	}

	/**
	 * @param int $orderId
	 *
	 * @return Invoice|null
	 * @throws NonUniqueResultException
	 */
	public function getOneByOrder(int $orderId): ?Invoice
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('i')
			->from(Invoice::class, 'i')
			->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
			->where('o.id = :orderId')
			->setParameter('orderId', $orderId);

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param int $invoiceId
	 *
	 * @return bool
	 */
	public function regenerateInvoice(int $invoiceId): bool
	{
		try {
			$this->em->beginTransaction();

			$qb = $this->em->createQueryBuilder();
			$qb->select('o')
				->from(Order::class, 'o')
				->join('o.invoice', 'i')
				->where('i.id = :invoiceId')
				->setParameter('invoiceId', $invoiceId);

			/** @var Order|null $order */
			$order = $qb->getQuery()->getOneOrNullResult();

			if ($order === null) {
				return false;
			}

			$invoice = $order->getInvoice();

			if ($invoice === null) {
				return false;
			}

			// --------- from here the data editing begins ---------

			$invoiceData = $invoice->invoiceData;

			// invoice payment
			$invoicePayment        = $invoiceData->payment;
			$invoicePayment->name  = $order->getPayment()->getName();
			$invoicePayment->price = $order->getPayment()->getPrice();

			// invoice spedition
			$invoiceSpedition        = $invoiceData->spedition;
			$invoiceSpedition->name  = $order->getSpedition()->getName();
			$invoiceSpedition->price = $order->getSpedition()->getPrice();

			// customer address
			$orderCustomerAddress     = $order->getAddressInvoice();
			$customerAddress          = $invoiceData->customer->address;
			$customerAddress->city    = $orderCustomerAddress->getCity();
			$customerAddress->country = $orderCustomerAddress->getCountry() ? $orderCustomerAddress->getCountry()->getName() : null;
			$customerAddress->postal  = $orderCustomerAddress->getPostal();
			$customerAddress->street  = $orderCustomerAddress->getStreet();

			// customer
			$invoiceCustomer            = $invoiceData->customer;
			$invoiceCustomer->phone     = $orderCustomerAddress->getPhone();
			$invoiceCustomer->email     = $orderCustomerAddress->getEmail();
			$invoiceCustomer->company   = $orderCustomerAddress->getCompany();
			$invoiceCustomer->firstName = $orderCustomerAddress->getFirstName();
			$invoiceCustomer->lastName  = $orderCustomerAddress->getLastName();
			$invoiceCustomer->idNumber  = $orderCustomerAddress->getIdNumber();
			$invoiceCustomer->vatNumber = $orderCustomerAddress->getVatNumber();

			if ($seller = $this->sellers->getSellerForSite($order->site->getIdent())) {
				/** @var Seller $seller */
				// bank
				$bankAccount       = $seller->getBankAccount();
				$bank              = $invoiceData->getSupplier()->getBank();
				$bank->bankAccount = $bankAccount->numberPart1;
				$bank->bankCode    = $bankAccount->numberPart2;
				$bank->iban        = $bankAccount->iban;
				$bank->swift       = $bankAccount->swift;

				// supplier address
				$supplierAddress         = $invoiceData->getSupplier()->getAddress();
				$supplierAddress->street = $seller->street;
				$supplierAddress->postal = $seller->postal;
				$supplierAddress->city   = $seller->city;

				// supplier
				$supplier             = $invoiceData->getSupplier();
				$supplier->email      = $seller->email;
				$supplier->vatNumber  = $seller->dic;
				$supplier->idNumber   = $seller->ic;
				$supplier->name       = $seller->name;
				$supplier->isPayerVat = $seller->dic ? true : false;
			} else {
				// bank
				$bank              = $invoiceData->getSupplier()->getBank();
				$bank->bankAccount = EshopOrdersConfig::load('invoice.bank.bankAccount');
				$bank->bankCode    = EshopOrdersConfig::load('invoice.bank.bankCode');
				$bank->iban        = EshopOrdersConfig::load('invoice.bank.iban');
				$bank->swift       = EshopOrdersConfig::load('invoice.bank.swift');

				// supplier address
				$supplierAddress         = $invoiceData->getSupplier()->getAddress();
				$supplierAddress->street = EshopOrdersConfig::load('invoice.supplier.street');
				$supplierAddress->postal = EshopOrdersConfig::load('invoice.supplier.postCode');
				$supplierAddress->city   = EshopOrdersConfig::load('invoice.supplier.city');

				// supplier
				$supplier             = $invoiceData->getSupplier();
				$supplier->email      = EshopOrdersConfig::load('invoice.supplier.email');
				$supplier->vatNumber  = EshopOrdersConfig::load('invoice.supplier.taxIdentificationNumber');
				$supplier->idNumber   = EshopOrdersConfig::load('invoice.supplier.companyIdentificationNumber');
				$supplier->name       = EshopOrdersConfig::load('invoice.supplier.name');
				$supplier->isPayerVat = EshopOrdersConfig::load('invoice.isPayerVat');
			}

			// remove products
			foreach ($invoiceData->products as $product) {
				$this->em->remove($product);
			}

			// actual products
			$invoiceProducts = [];
			foreach ($order->getOrderItems() as $orderItem) {
				$dbProduct      = $this->products->get($orderItem->getProductId());
				$invoiceProduct = new Invoice\Product($invoiceData);

				if ($dbProduct->getGallery() && $dbProduct->getGallery()->getCoverImage() && file_exists($dbProduct->getGallery()->getCoverImage()->getFile())) {
					$invoiceProduct->imageBase64 = ImageHelper::createBase64Miniature(
						$dbProduct->getGallery()->getCoverImage()->getFile(),
						EshopOrdersConfig::load('invoice.productMiniature.width'),
						null,
						Image::SHRINK_ONLY
					);
				}

				$invoiceProduct->name      = $dbProduct->getProductText($invoiceData->lang)->name ?? null;
				$invoiceProduct->productId = $dbProduct->getId();
				$invoiceProduct->vatRate   = $orderItem->getVatRate();
				$invoiceProduct->price     = $orderItem->getPrice();
				$invoiceProduct->vatRate   = (int) $orderItem->getVatRate();
				$invoiceProduct->quantity  = $orderItem->getQuantity();

				$invoiceProducts[] = $invoiceProduct;
			}

			$invoiceData->products = $invoiceProducts;

			// remove discounts
			foreach ($invoiceData->discounts as $discount) {
				$this->em->remove($discount);
			}

			// actual discounts
			$invoiceDiscounts = [];
			$currentPrice     = $invoiceData->getPriceItems();
			foreach ($order->getOrderDiscounts() as $orderDiscount) {
				$amount                 = $orderDiscount->calculateDiscount($currentPrice);
				$currentPrice           += $amount;
				$invoiceDiscount        = new Invoice\Discount($invoiceData);
				$invoiceDiscount->price = $amount;
				$invoiceDiscount->name  = $orderDiscount->getName();
				$invoiceDiscount->type  = $orderDiscount->getType();
				$invoiceDiscount->value = $orderDiscount->getValue();
				$invoiceDiscount->code  = $orderDiscount->getCode();

				$invoiceDiscounts[] = $invoiceDiscount;
			}

			$invoiceData->discounts = $invoiceDiscounts;

			$this->eventDispatcher->dispatch(new InvoiceRegenerateEvent($invoice), 'eshopOrders.beforeSaveRegeneratedInvoice');

			$this->em->persist($invoiceData);
			$this->em->flush();

			$this->em->commit();

			return true;
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			return false;
		}
	}

	/**
	 * @param array $getSitesIdent
	 *
	 * @return int
	 * @throws NoResultException
	 * @throws NonUniqueResultException
	 */
	public function getIvoicesCount(array $getSitesIdent): int
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('COUNT(i.id)')
			->from(Invoice::class, 'i')
			->join('i.site', 's')
			->where($qb->expr()->in('s.ident', $getSitesIdent));

		return (int) $qb->getQuery()->getSingleScalarResult();
	}

}
