<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemGift;
use Nette\Localization\ITranslator;

/**
 * class Orders
 * @package EshopOrders\Model
 *
 * @method Order|object|null getReference($id)
 * @method Order[]|null getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{

	protected $entityClass = Order::class;

	/** @var ITranslator @inject */
	public $translator;

	/** Vrati seznam objednavek podle zakaznika
	 * @return ArrayCollection|Order[]
	 */
	public function getByCustomer($customer = null)
	{
		if (is_null($customer)) {
			return new ArrayCollection();
		}

		$orderRawQuery = $this->getEr()->createQueryBuilder('o', 'o.id');
		$orderRawQuery->andWhere('o.customer = :customer')->setParameter('customer', $customer);
		$orderRawQuery->orderBy('o.id', 'DESC');

		return $orderRawQuery->getQuery()->getResult();
	}

	public function getByIdent($ident)
	{
		if (is_null($ident)) {
			return null;
		}
		$orderQuery = $this->getEr()->createQueryBuilder('o', 'o.id');
		$orderQuery->andWhere('o.ident = :ident')->setParameter('ident', $ident);

		return $orderQuery->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param CartItem[] $itemsCart
	 * @param            $order
	 *
	 * @return array
	 * @throws \Doctrine\ORM\ORMException
	 *
	 */
	public function fillOrderItems($itemsCart, $order)
	{
		$items = [];
		foreach ($itemsCart as $ci) {
			$product = $ci->getProduct();
			$item    = new OrderItem($this->em->getReference(Product::class, $product->getId()), $order);

			$item->setVariantId($ci->getVariantId());
			$item->setQuantity($ci->getQuantity());
			$item->setPrice($product->priceInBaseCurrency);

			if ($product) {
				$item->setVatRate($product->getVatRate());
				$item->setCode1($product->getCode1());
			}

			$item->addOrderItemText($this->translator->getLocale());
			$item->getOrderItemText($this->translator->getLocale())->setName($ci->title);

			$variantName = '';
			if ($ci->getProductVariant()) {
				$variants    = $ci->getProductVariant()->variants;
				$variantName = '';

				end($variants);
				$last_key = key($variants);
				reset($variants);

				foreach ($variants as $key => $variant) {
					$variantName .= $variant['variantName'] . ': ' . $variant['variantValueName'];
					if ($key != $last_key) {
						$variantName .= ', ';
					}
				}
			}
			$item->getOrderItemText($this->translator->getLocale())->setVariantName($variantName);

			if ($ci->discountDisabled)
				$item->setMoreDataValue('discountDisabled', true);

			foreach ($ci->getGifts() as $gift) {
				// TODO možno upravit protože OrderItem by měl obshovat referenci na produkt
				$itemGift        = new OrderItemGift($item, $this->em->getReference(Product::class, $gift->getProductId()), $gift->getName());
				$itemGift->code1 = $gift->code1;
				$itemGift->ean   = $gift->ean;

				$this->em->persist($itemGift);
			}

			$items[] = $item;
		}

		return $items;
	}
}
