<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Invoice\Discount;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Statuses;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Nette\Utils\Strings;
use Tracy\Debugger;

class CorrectiveTaxDocumentForm extends BaseControl
{
	protected Order $order;
	protected Invoices $invoices;
	protected Statuses $statusesService;
	protected Payments $payments;

	/**
	 * CorrectiveTaxDocumentForm constructor.
	 * @param Order $order
	 * @param Invoices $invoices
	 * @param Statuses $statusesService
	 */
	public function __construct(Order $order, Invoices $invoices, Statuses $statusesService, Payments $payments)
	{
		$this->order = $order;
		$this->invoices = $invoices;
		$this->statusesService = $statusesService;
		$this->payments = $payments;
	}

	public function render(): void
	{
		$this->template->order = $this->order;
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			 ->setShowLangSwitcher(false);

		/** @var OrderItem $oi */
		foreach ($this->order->getOrderItems()->toArray() as $oi) {
			$container = $form->addContainer('prod_' . $oi->getId());
			$container->addHidden('oiId', $oi->getId());
			$container->addText('price')
					  ->setDefaultValue($oi->getPrice())
					  ->addCondition($form::FILLED)
					  ->addRule($form::FLOAT)
					  ->addRule($form::MIN, null, 0)
					  ->addRule($form::MAX, null, $oi->getPrice());
			$container->addInteger('quantity')
					  ->setDefaultValue($oi->getQuantity())
					  ->addCondition($form::FILLED)
					  	->addRule($form::MIN, null, 1)
					  	->addRule($form::MAX, null, $oi->getQuantity());
		}

		$orderDiscounts = $this->order->getOrderDiscounts()->toArray();
		usort($orderDiscounts, fn(OrderDiscount $o1, OrderDiscount $o2) => $o1->getSortPosition() <=> $o2->getSortPosition());
		/** @var OrderDiscount $od */
		foreach ($orderDiscounts as $od) {
			$container = $form->addContainer('discount_' . $od->getId());
			$container->addHidden('odId', $od->getId());
			$container->addText('price')
					  ->setDefaultValue(abs($od->getPrice(true)))
					  ->addCondition($form::FILLED)
					  ->addRule($form::FLOAT)
					  ->addRule($form::MIN, null, 0);
		}

		$form->addText('paymentPrice', 'eshopOrders.correctiveTaxDocumentForm.paymentPrice')
			 ->setDefaultValue($this->order->getPayment()->getPrice())
			 ->setRequired()
			 ->addRule($form::NUMERIC)
			 ->addRule($form::MIN, null, 0);
		$form->addText('speditionPrice', 'eshopOrders.correctiveTaxDocumentForm.speditionPrice')
			 ->setDefaultValue($this->order->getSpedition()->getPrice())
			 ->setRequired()
			 ->addRule($form::NUMERIC)
			 ->addRule($form::MIN, null, 0);

		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 */
	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$vals = $form->getHttpData();
		$hasProducts = Arrays::some(array_keys($vals), fn($inputKey) => Strings::startsWith($inputKey, 'prod_'));
		if (!$hasProducts) {
			$form->addError('eshopOrders.correctiveTaxDocumentForm.errors.fillProducts');
			return;
		}
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->em->beginTransaction();

		try {
			$products = [];
			foreach ($form->getHttpData() as $k => $container) {
				if (!Strings::startsWith($k, 'prod_')) {
					continue;
				}
				$productId = (int) explode('_', $k)[1];
				$products[$productId] = $container;
			}
			$orderDiscounts = [];
			foreach ($form->getHttpData() as $k => $container) {
				if (!Strings::startsWith($k, 'discount_')) {
					continue;
				}
				$discountId = (int) explode('_', $k)[1];
				$orderDiscounts[$discountId] = $container;
			}

			$order = clone $this->order;
			$orderItems = [];
			/** @var OrderItem $item */
			foreach ($this->order->getOrderItems()->toArray() as $item) {
				if (in_array($item->getId(), array_keys($products))) {
					$newOi = clone $item;
					$newOi->setQuantity((int) $products[$item->getId()]['quantity']);
					$newOi->setPrice(-((float) $products[$item->getId()]['price']));
					$oit = clone $item->getOrderItemText();
					$oit->setId($newOi);
					$this->em->persist($oit);
					$newOi->setOrderItemText($oit);
					$newOi->order = $order;
					$orderItems[] = $newOi;
				}
			}

			$order->setOrderItems($orderItems);

			$orderCurrency = null;
			if ($this->order->currency) {
				$oc = $this->order->currency;
				$orderCurrency = new OrderCurrency($order, $oc->code);
				$orderCurrency->rate = $oc->rate;
				$orderCurrency->decimals = $oc->decimals;
			}
	
			$op = $this->order->getPayment();
			$orderPayment = clone $op;

			$paymentIdent = EshopOrdersConfig::load('correctiveTaxDocument.payment');
			if ($paymentIdent) {
				/** @var Payment|null $payment */
				$payment = $this->payments->getEr()->findOneBy(['ident' => $paymentIdent], ['id' => 'asc']);
				if ($payment) {
					$orderPayment->setName($payment->getName());
					$orderPayment->setPayment($payment);
				}
			}

			$orderPayment->setPrice(-((float) $values->paymentPrice));
			$orderPayment->order = $order;

			$os = $this->order->getSpedition();
			$orderSpedition = clone $os;
			$orderSpedition->setPrice(-((float) $values->speditionPrice));
			$orderSpedition->order = $order;

			$ai = $this->order->getAddressInvoice();
			$addressInvoice = clone $ai;

			$ad = $this->order->getAddressDelivery();
			$addressDelivery = clone $ad;

			$hasStatusCreated = false;
			$hasStatusSpedition = false;
			$hasStatusFinished = false;
			foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
				if ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_CREATED) {
					$hasStatusCreated = true;
				} elseif ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_SPEDITION) {
					$hasStatusSpedition = true;
				} elseif ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_FINISHED) {
					$hasStatusFinished = true;
				}
			}

			if (!$hasStatusCreated) {
				$statusCreated = $this->statusesService->get(OrderStatus::STATUS_CREATED);
				$orderActionCreated = new OrderStatus($order, $statusCreated);
				$this->em->persist($orderActionCreated);
			}

			if (!$hasStatusSpedition) {
				$statusSpedition = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
				$orderActionSpedition = new OrderStatus($order, $statusSpedition);
				$this->em->persist($orderActionSpedition);
			}

			if (!$hasStatusFinished) {
				$statusFinished = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
				$orderActionFinished = new OrderStatus($order, $statusFinished);
				$this->em->persist($orderActionFinished);
			}

			$order->isCorrectiveTaxDocument = 1;
			$order->correctiveTaxDocumentOf = $this->order;
			$order->isPaid = 1;
			$order->currency = $orderCurrency;
			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);
			$order->setAddressInvoice($addressInvoice);
			$order->setAddressDelivery($addressDelivery);
			$order->setInvoice(null);

			/** @var OrderItem $oi */
			foreach ($order->getOrderItems()->toArray() as $oi) {
				$this->em->persist($oi);
			}
			$orderStatuses = [];
			/** @var OrderStatus $os */
			foreach ($order->getOrderStatuses()->toArray() as $item) {
				$os = clone $item;
				$os->setOrder($order);
				$this->em->persist($os);
			}
			$order->setOrderStatuses($orderStatuses);

			$newOrderDiscounts = [];
			foreach ($orderDiscounts as $id => $orderDiscount) {
				/** @var OrderDiscount $od */
				foreach ($order->getOrderDiscounts()->toArray() as $od) {
					if ($od->getId() === (int) $orderDiscount['odId']) {
						$newOrderDiscount = clone $od;

						$amount = -abs((float) $orderDiscount['price']);
						$newOrderDiscount->setPrice($amount);
						$newOrderDiscount->setOrder($order);
						$this->em->persist($newOrderDiscount);
						$newOrderDiscounts[] = $newOrderDiscount;
					}
				}
			}

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			$this->em->persist($addressInvoice);
			$this->em->persist($addressDelivery);
			$this->em->flush();

			$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);
			$order->setInvoice($invoice);

			foreach ($invoice->invoiceData->discounts->toArray() as $od) {
				$this->em->remove($od);
			}

			$invoice->invoiceData->discounts->clear();
			foreach ($newOrderDiscounts as $orderDiscount) {
				$newDiscount = new Discount($invoice->invoiceData);
				$newDiscount->name = $orderDiscount->getName();
				$newDiscount->type = $orderDiscount->getType();
				$newDiscount->code = $orderDiscount->getCode();
				$newDiscount->value = $orderDiscount->getValue();
				$newDiscount->price = $orderDiscount->getPrice();

				$this->em->persist($newDiscount);
			}

			$this->em->flush();

			$event = new OrderEvent($order);
			$event->orderPayment = $orderPayment;
			$event->orderSpedition = $orderSpedition;
			$event->addrDelivery = $addressDelivery;
			$event->addrInvoice = $addressInvoice;
			$event->orderItems = $orderItems;

			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');

			$this->em->commit();

			$this->presenter->flashMessageSuccess('eshopOrders.correctiveTaxDocumentForm.created');
		} catch (\Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$this->presenter->flashMessageDanger('eshopOrders.correctiveTaxDocumentForm.savedError');
		}
	}

}