<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Core\Model\Application\AppState;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Core\Model\Event\EventDispatcher;
use Core\Model\Entities\Site;
use Core\Model\Sites;
use EshopOrders\ApiModule\Api\V1\Model\Orders;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderPayloadReturnEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Receipts;
use EshopOrders\Model\Speditions;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Utils\Helpers;
use EshopStock\Model\Repository\SupplyProducts;
use Apitte\Core\Annotation\Controller\RequestBody;
use Nette\Utils\DateTime;
use Tracy\Debugger;


/**
 * @Path("/orders")
 */
class OrdersController extends BaseController
{
	protected EventDispatcher        $eventDispatcher;
	protected EntityManagerDecorator $em;
	protected Sites                  $sitesService;
	protected Payments               $paymentsService;
	protected Speditions             $speditionsService;
	protected Statuses               $statusesService;
	protected Customers              $customersService;
	protected Invoices               $invoices;
	protected Orders                 $orders;
	protected Receipts               $receipts;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		EntityManagerDecorator $em,
		Sites                  $sitesService,
		Payments               $paymentsService,
		Speditions             $speditionsService,
		Statuses               $statusesService,
		Customers              $customersService,
		Invoices               $invoices,
		Orders                 $orders,
		Receipts               $receipts
	)
	{
		$this->em                = $em;
		$this->sitesService      = $sitesService;
		$this->paymentsService   = $paymentsService;
		$this->speditionsService = $speditionsService;
		$this->statusesService   = $statusesService;
		$this->customersService  = $customersService;
		$this->eventDispatcher   = $eventDispatcher;
		$this->invoices          = $invoices;
		$this->orders            = $orders;
		$this->receipts          = $receipts;
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getStatusesOverview()));
	}

	/**
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function getDetail(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['order' => $this->orders->getDetail($request->getParameter('id'))]));
	}

	/**
	 * @Path("/status/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatus($request->getParameter('status'))));
	}

	/**
	 * @Path("/status-not/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatusNot(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatusNot($request->getParameter('status'))));
	}

	/**
	 * @Path("/{id}/status")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus")
	 */
	public function setOrderStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->setOrderStatus(
					$request->getParameter('id'),
					$request->getEntity(),
				),
			]));
	}

	/**
	 * @Path("/{id}/payment")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Payment")
	 */
	public function changePayment(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changePayment(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/{id}/spedition")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition")
	 */
	public function changeSpedition(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changeSpedition(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/create")
	 * @Method("POST")
	 */
	public function create(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();

		try {
			$this->em->getConnection()->beginTransaction();

			$order = new Order($this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent()));
			$order->setMessage($data['globalNote'] ?? '');
			$order->setAgreedTerms(true);

			AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.checkout');
			$order->isPaid = 1;
			$order->paid   = new DateTime;
			$orderItems    = $this->orders->convertJSONProductsToOrderItems($data['checkoutProducts'], $order);
			$isCorrective  = Helpers::isCorrectiveByOrderItems($orderItems);
			$order->setOrderItems($orderItems);
			$order->isCorrectiveTaxDocument = (int) $isCorrective;

			$orderPayment   = new OrderPayment($this->paymentsService->getReferenceByIdent($data['payment']), $order);
			$orderSpedition = new OrderSpedition($this->speditionsService->getReferenceByIdent($data['spedition']), $order);

			$statusCreated        = $this->statusesService->get(OrderStatus::STATUS_CREATED);
			$orderActionCreated   = new OrderStatus($order, $statusCreated);
			$statusSpedition      = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
			$orderActionSpedition = new OrderStatus($order, $statusSpedition);
			$statusFinished       = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
			$orderActionFinished  = new OrderStatus($order, $statusFinished);

			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);
			$order->setOrderStatuses([$orderActionCreated, $orderActionFinished]);

			$addressInvoice  = $this->orders->createAddress(new OrderAddress(OrderAddress::ADDRESS_INVOICE), $data['customer']['addressInvoice']);
			$addressDelivery = $this->orders->createAddress(new OrderAddress(OrderAddress::ADDRESS_DELIVERY), $data['customer']['addressDelivery']);
			$order->setAddressInvoice($addressInvoice);
			$order->setAddressDelivery($addressDelivery);

			if ($data['customer']['customerId'] && $customer = $this->customersService->get((int) $data['customer']['customerId'])) {
				$order->setCustomer($customer);
				$this->em->persist($customer);
			}

			$this->orders->addGlobalSales($data['globalSales'], $order, $data['totalPrice']);

			/*if ($data['customer']['customerCreated'] && ($data['customer']['delivery']['email'] || $data['customer']['invoice']['email'])) {
				// Id vkladam jenom formalne - nemel by existovat
				$customer = $this->customersService->getOrCreateCustomer($data['customer']['customerId']);
				$customer->setAddressDelivery($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$customer->setAddressInvoice($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$this->em->persist($customer);
			}*/

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $orderPayment;
			$event->orderSpedition = $orderSpedition;
			$event->addrDelivery   = $addressDelivery;
			$event->addrInvoice    = $addressInvoice;
			$event->orderItems     = $orderItems;
			$event->formData       = $data;

			$eshopStockSupplyProduct = class_exists('EshopStock\DI\EshopStockExtension') ? SupplyProducts::class : null;

			if ($eshopStockSupplyProduct !== null) {
				$eshopStockSupplyProduct::$assignOrderItemFlush = false;
			}

			$this->eventDispatcher->dispatch($event, 'eshopCheckout.orderBeforeSave');

			if (EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) && in_array($orderPayment->getIdent(), ['storeCard',
					'storeCash'])) {
				$order->receiptIdent = $this->receipts->generateIdent($order);
			}

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			$this->em->persist($orderActionCreated);
			$this->em->persist($orderActionSpedition);
			$this->em->persist($orderActionFinished);

			$this->em->flush();

			if ($isCorrective) {
				$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);
				$order->setInvoice($invoice);
				$this->em->flush();
				$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');
			}

			$this->em->getConnection()->commit();

			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.orderOnSuccess');

			$result = [
				'success' => $order,
			];

			$orderPayloadReturnEvent = new OrderPayloadReturnEvent($result, $order);
			$this->eventDispatcher->dispatch($orderPayloadReturnEvent, 'eshopOrders.api.orderPayloadReturnEvent');

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($orderPayloadReturnEvent->data));
		} catch (\Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
				'data'  => $data,
			]));
		}
	}
}
