<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Model;

use Apitte\Core\Exception\Api\ClientErrorException;
use Apitte\Core\Exception\ApiException;
use Apitte\Core\Http\ApiResponse;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Sites;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus as BodyEntityOrderStatus;
use EshopOrders\ApiModule\Api\V1\BodyEntity\Payment;
use EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition;
use EshopOrders\ApiModule\Api\V1\Exceptions\ErrorException;
use EshopOrders\ApiModule\Api\V1\Model\Dao\Address;
use EshopOrders\ApiModule\Api\V1\Model\Helpers\CustomerHelper;
use EshopOrders\FrontModule\Model\Orders as FrontOrders;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\Statuses;
use Nette\Localization\Translator;
use EshopOrders\ApiModule\Api\V1\Model\Dao\OrderPayment;
use EshopOrders\ApiModule\Api\V1\Model\Dao\OrderSpedition;
use Tracy\Debugger;

class Orders
{
	protected EntityManagerDecorator $em;
	protected Translator             $translator;
	protected Countries              $countries;
	protected EventDispatcher        $eventDispatcher;
	protected Statuses               $statuses;
	protected FrontOrders            $frontOrders;
	protected CustomerHelper         $customerHelper;

	public function __construct(
		EntityManagerDecorator $em,
		Translator             $translator,
		Countries              $countries,
		Statuses               $statuses,
		EventDispatcher        $eventDispatcher,
		FrontOrders            $frontOrders,
		CustomerHelper         $customerHelper
	)
	{
		$this->em              = $em;
		$this->translator      = $translator;
		$this->countries       = $countries;
		$this->statuses        = $statuses;
		$this->eventDispatcher = $eventDispatcher;
		$this->frontOrders     = $frontOrders;
		$this->customerHelper  = $customerHelper;
	}

	public function getDetail(int $id): ?Dao\Order
	{
		$orderEntity = $this->frontOrders->getDetail($id);

		if (!$orderEntity) {
			throw new ErrorException("Order '$id' not found", ApiResponse::S404_NOT_FOUND);
		}

		$order = new Dao\Order(
			$orderEntity->getId(),
			$orderEntity->getIdent(),
			$orderEntity->site->getIdent(),
			$orderEntity->lang,
			$orderEntity->getPrice(),
			$orderEntity->getPriceWithoutVat(),
		);

		$customer = $orderEntity->getCustomer();
		if ($customer) {
			$order->customerId = $customer->getId();
			$order->phone      = $customer->getPhone();

			$user             = $customer->getUser();
			$order->firstName = $user->name;
			$order->lastName  = $user->lastname;
			$order->email     = $user->email;

			if ($orderEntity->getCustomer()->getGroupCustomers()) {
				$order->customerGroupId = $orderEntity->getCustomer()->getGroupCustomers()->getId();
			}
		}

		$order->agreedTerms = (bool) $orderEntity->getAgreedTerms();
		$order->paid        = $orderEntity->paid;
		$order->message     = $orderEntity->getMessage();

		$order->deliveryAddress = $this->customerHelper->fillAddressByOrderAddress($orderEntity->getAddressDelivery(), new Address());
		$order->invoiceAddress  = $this->customerHelper->fillAddressByOrderAddress($orderEntity->getAddressInvoice(), new Address());

		foreach ($orderEntity->getOrderItems() as $item) {
			$orderItem = new Dao\OrderItem(
				$item->getId(),
				$item->getOrderItemText($order->lang)->getName(),
				$item->getQuantity(),
				$item->getPrice(),
				$item->getPriceWithoutVat(),
			);

			$orderItem->productId    = $item->getProductId();
			$orderItem->code1        = $item->getCode1();
			$orderItem->ean          = $item->getProduct() ? $item->getProduct()->ean : null;
			$orderItem->recyclingFee = $item->getRecyclingFeeWithoutVat();
			$orderItem->vatRate      = $item->getVatRate();
			$orderItem->moreData     = (array) $item->getMoreData();
			$orderItem->uploads      = $item->getUploadedFiles();

			$order->items[] = $orderItem;
		}

		foreach ($orderEntity->getOrderDiscounts() as $item) {
			$orderDiscount = new Dao\OrderDiscount(
				$item->getValue(),
				$item->getPrice(true),
				$item->getType()
			);

			$orderDiscount->code = $item->getCode();
			$orderDiscount->name = $item->getName();

			$order->discounts[] = $orderDiscount;
		}

		foreach ($orderEntity->getOrderStatuses() as $item) {
			$orderStatus = new Dao\OrderStatus(
				$item->getStatus()->getId(),
				$item->getCreated()
			);

			$orderStatus->parameter = $item->getParameter();
			$orderStatus->message   = $item->getMessage();

			$order->statuses[$item->getStatus()->getId()] = $orderStatus;
		}

		foreach ($orderEntity->getOrderFlags() as $item) {
			$orderFlag = new Dao\OrderFlag(
				$item->getType(),
				(bool) $item->getState(),
			);

			$order->flags[$item->getType()] = $orderFlag;
		}

		foreach ($orderEntity->getGifts() as $item) {
			$orderGift = new Dao\OrderGift(
				$item->getName()
			);

			$orderGift->product = $item->getProductId();
			$orderGift->code1   = $item->code1;
			$orderGift->ean     = $item->ean;

			$order->gifts[] = $orderGift;
		}

		if ($orderEntity->getPayment()) {
			$payment        = new OrderPayment();
			$payment->id    = $orderEntity->getPayment()->getPaymentId();
			$payment->ident = $orderEntity->getPayment()->getIdent();
			$payment->name  = $orderEntity->getPayment()->getName();
			$payment->price = $orderEntity->getPayment()->getPrice();

			$order->payment = $payment;
		}

		if ($orderEntity->getSpedition()) {
			$spedition        = new OrderSpedition();
			$spedition->id    = $orderEntity->getSpedition()->getSpeditionId();
			$spedition->ident = $orderEntity->getSpedition()->getIdent();
			$spedition->name  = $orderEntity->getSpedition()->getName();
			$spedition->price = $orderEntity->getSpedition()->getPrice();

			$order->spedition = $spedition;
		}

		return $order;
	}

	public function getIdByStatus(string $status): array
	{
		return array_map(
			static fn($row) => $row['id'],
			$this->em->getConnection()->fetchAllAssociative("SELECT s.order_id as id, s.status_id as status FROM eshop_orders__order_status s
			WHERE s.created = (SELECT MAX(s2.created) FROM eshop_orders__order_status s2 WHERE s.order_id = s2.order_id)
			AND s.status_id = '{$status}'
			ORDER BY s.order_id DESC")
		);
	}

	public function getIdByStatusNot(string $status): array
	{
		return array_map(
			static fn($row) => $row['id'],
			$this->em->getConnection()->fetchAllAssociative("SELECT s.order_id as id, s.status_id as status FROM eshop_orders__order_status s
			WHERE s.created = (SELECT MAX(s2.created) FROM eshop_orders__order_status s2 WHERE s.order_id = s2.order_id)
			AND s.status_id NOT IN ('" . implode("', '", explode(',', $status)) . "')
			ORDER BY s.order_id DESC")
		);
	}

	public function setOrderStatus(int $orderId, BodyEntityOrderStatus $orderStatus): bool
	{
		/** @var Status|null $status */
		$status = $this->em->getRepository(Status::class)->find($orderStatus->status);
		if (!$status) {
			Debugger::log("Status not found - order '{$orderId} status '{$orderStatus->status}", 'api-eshoporders-order');

			return false;
		}

		/** @var Order|null $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);
		if (!$order) {
			Debugger::log("Order not found - order '{$orderId} status '{$orderStatus->status}", 'api-eshoporders-order');

			return false;
		}

		try {
			$newStatus = new OrderStatus(
				$order,
				$status,
			);

			if ($orderStatus->message) {
				$newStatus->setMessage($orderStatus->message);
			}

			$this->em->persist($newStatus);
			$this->em->flush();
		} catch (\Exception $e) {
			Debugger::log("Change status failed - order '{$orderId} status '{$statusId}", 'api-eshoporders-order');

			return false;
		}

		if ($orderStatus->sendToCustomer) {
			$this->statuses->sendOrderStatusEmail($order, $newStatus);
		}

		return true;
	}

	public function getStatusesOverview(): array
	{
		$result = [];

		$usedIds = [];
		$qb      = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('s')
			->select('IDENTITY(s.order) as order, IDENTITY(s.status) as status')
			->where('s.deleted IS NULL')
			->orderBy('s.created', 'DESC');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			if (isset($usedIds[$row['order']])) {
				continue;
			}

			$result[$row['status']][] = $row['order'];
			$usedIds[$row['order']]   = $row['order'];
		}
		$usedIds = null;

		foreach ($result as $status => $ids) {
			$result[$status] = array_reverse($ids);
		}

		return $result;
	}

	public function convertJSONProductsToOrderItems(array $JSONproducts, Order $order): array
	{
		$result       = [];
		$isCorrective = false;
		foreach ($JSONproducts as $item) {
			$product = $this->em->getReference(Product::class, $item['id']);
			$lang    = $this->translator->getLocale();
			$name    = $item['name'];

			if (EshopCatalogConfig::load('addManufacturerBeforeProductName', false) && $product->getManufacturer()) {
				$name = trim($product->getManufacturer()->name . ' ' . $name);
			}

			$price     = (float) $item['price'];
			$quantity  = (int) $item['count'];
			$orderItem = new OrderItem($product, $order);
			$orderItem->setQuantity(abs($quantity));
			$orderItem->addOrderItemText($lang);
			$orderItem->getOrderItemText($lang)->setName($name);
			if ($quantity < 0 || $isCorrective) { // pokud ma alespon jeden produkt zapornou kvantitu, pak se objednavka stava oprav. dan. dokladem
				$price        = -$price;
				$isCorrective = true;
			}

			$orderItem->setPrice($price);
			$orderItem->recyclingFee = $product->recyclingFee;
			$orderItem->setVatRate($item['vatRate']);
			$orderItem->setCode1($item['code1']);

			$this->addOrderItemsSales($item, $orderItem);

			$this->em->persist($orderItem);
			$result[] = $orderItem;
		}

		return $result;
	}

	public function addGlobalSales(array $globalSales, Order $order, $totalPrice): void
	{
		foreach ($globalSales as $sale) {
			$saleEntity = new OrderDiscount('CHECKOUTSALE', $order);
			$saleEntity->setName($sale['name']);

			if (!empty($sale['saleCode'])) {
				$saleEntity->setCode($sale['saleCode']);
			}

			// procentualni sleva
			if ($sale['percentSale'] > 0) {
				$saleEntity->setValue((float) $sale['percentSale']);
				$saleEntity->setType(OrderDiscount::TYPE_PERCENT);
				$price = (100 / (100 - (float) $sale['percentSale'])) * $totalPrice; // cena pred uplatneni slevy
				$saleEntity->setPrice($price - $totalPrice);
			}

			// pevna sleva
			if ($sale['staticSale'] > 0) {
				$saleEntity->setValue((float) $sale['staticSale']);
				$saleEntity->setType(OrderDiscount::TYPE_FIX);
				$saleEntity->setPrice((float) $sale['staticSale']);
			}

			$order->addOrderDiscount($saleEntity);
			$this->em->persist($saleEntity);
		}
	}

	/**
	 * @param $item
	 * @param $orderItem
	 */
	public function addOrderItemsSales($item, $orderItem): void
	{

		// Pevna sleva na vsechny kusy
		if (($staticSale = $item['sale']['all']['staticSale']) > 0) {
			$sale = new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ALL);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na vsechny kusy
		if (($percentSale = $item['sale']['all']['percentSale']) > 0) {
			$sale = new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ALL);
			$this->em->persist($sale);
		}

		// Pevna sleva na kus
		if (($staticSale = $item['sale']['one']['staticSale']) > 0) {
			$sale = new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ONE);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na kus
		if (($percentSale = $item['sale']['one']['percentSale']) > 0) {
			$sale = new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ONE);
			$this->em->persist($sale);
		}
	}

	public function createAddress(OrderAddress $address, array $addressData): OrderAddress
	{

		$address->setFirstName($addressData['firstName'] ?? '');
		$address->setLastName($addressData['lastName'] ?? '');
		$address->setEmail($addressData['email'] ?? '');
		$address->setPhone($addressData['phone'] ?? '');
		$address->setStreet($addressData['street'] ?? '');
		$address->setCity($addressData['city'] ?? '');
		$address->setPostal($addressData['postal'] ?? '');
		if (!empty($addressData['country'])) {
			$address->setCountry($this->countries->getReference($addressData['country']));
		}
		$address->setCompany($addressData['company'] ?? '');
		$address->setIdNumber($addressData['idNumber'] ?? '');
		$address->setVatNumber($addressData['vatNumber'] ?? '');
		$this->em->persist($address);
		bdump($address);

		return $address;
	}

	public function changePayment(int $orderId, Payment $data): ?array
	{
		/** @var ?Order $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);

		if (!$order) {
			return null;
		}

		try {
			if ($data->paymentId) {
				/** @var ?\EshopOrders\Model\Entities\Payment $payment */
				$payment = $this->em->getRepository(\EshopOrders\Model\Entities\Payment::class)->find($data->paymentId);

				if ($payment) {
					$orderPayment = $order->getPayment();

					if (!$orderPayment) {
						$orderPayment = new \EshopOrders\Model\Entities\OrderPayment(
							$payment,
							$order,
						);
						$orderPayment->setVatRate(21);
						$orderPayment->setPrice((float) $payment->getPrice());

						$order->setPayment($orderPayment);
					} else {
						$orderPayment->setPayment($payment);
					}

					$orderPayment->setName($payment->getName());

					$this->em->persist($orderPayment);
				} else {
					return null;
				}
			}

			if ($data->price !== null) {
				$orderPayment = $order->getPayment();
				if ($orderPayment) {
					$orderPayment->setPrice($data->price);
					$this->em->persist($orderPayment);
				}
			}

			$this->em->flush();

			return [];
		} catch (\Exception $e) {
			Debugger::log($e, Debugger::EXCEPTION);
		}

		return null;
	}

	public function changeSpedition(int $orderId, Spedition $data): ?array
	{
		/** @var ?Order $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);

		if (!$order) {
			return null;
		}

		try {
			if ($data->speditionId) {
				/** @var ?\EshopOrders\Model\Entities\Spedition $spedition */
				$spedition = $this->em->getRepository(\EshopOrders\Model\Entities\Spedition::class)->find($data->speditionId);

				if ($spedition) {
					$orderSpedition = $order->getSpedition();

					if (!$orderSpedition) {
						$orderSpedition = new \EshopOrders\Model\Entities\OrderSpedition(
							$spedition,
							$order,
						);
						$orderSpedition->setVatRate(21);
						$orderSpedition->setPrice((float) $spedition->getPrice());

						$order->setSpedition($orderSpedition);
					} else {
						$orderSpedition->setSpedition($spedition);
					}

					$orderSpedition->setName($spedition->getName());

					$this->em->persist($orderSpedition);
				} else {
					return null;
				}
			}

			if ($data->price !== null) {
				$orderSpedition = $order->getSpedition();
				if ($orderSpedition) {
					$orderSpedition->setPrice($data->price);
					$this->em->persist($orderSpedition);
				}
			}

			$this->em->flush();

			return [];
		} catch (\Exception $e) {
			Debugger::log($e, Debugger::EXCEPTION);
		}

		return null;
	}
}
