<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use Gallery\FrontModule\Model\Dao\Image;
use EshopCatalog\FrontModule\Model\Dao\Product;

class CartItem
{
	/** @var int|string */
	public $id;

	public ?string $ident = null;

	/** @var int|null */
	public $productId;

	/** @var int */
	public $quantity;

	/** @var Product|null */
	public $product;

	/** @var float */
	public $price;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var Image */
	public $image;

	/** @var string */
	public $link;

	/** @var string */
	public $title;

	/** @var int */
	protected $vatRate;

	/** @var bool */
	public $freeDelivery = false;

	/** @var bool */
	public $discountDisabled = false;

	/** @var array */
	protected $data = [];

	/** @var CartItemGift[] */
	protected $gifts = [];

	/** @var self[] */
	protected array $childs = [];

	public ?int $parentId = null;

	/**
	 * Grams
	 * @var int
	 */
	public int $weight = 0;

	public function __construct(string $title, ?int $productId, float $price)
	{
		$this->title     = $title;
		$this->productId = $productId;
		$this->price     = $price;
	}

	public function getId(): int { return $this->id; }

	public function setId(int $id): self
	{
		$this->id = $id;

		return $this;
	}

	public function getProductId(): ?int { return $this->productId; }

	public function setProductId(int $productId): self
	{
		$this->productId = $productId;

		return $this;
	}

	public function getQuantity(): int { return $this->quantity; }

	public function setQuantity(int $quantity): self
	{
		$this->quantity = $quantity;

		return $this;
	}

	public function getProduct(): ?Product { return $this->product; }

	public function setProduct(Product $product): self
	{
		$this->product = $product;

		return $this;
	}

	public function getImage(): ?Image { return $this->image; }

	public function setImage(Image $image): self
	{
		$this->image = $image;

		return $this;
	}

	public function getLink(): ?string { return $this->link; }

	public function setLink(string $link): self
	{
		$this->link = $link;

		return $this;
	}

	public function getPriceWithoutChilds(bool $useBaseCurrency = false): float
	{
		return $useBaseCurrency ? $this->priceInBaseCurrency : $this->price;
	}

	public function getPriceWithoutChildsWithoutVat(bool $useBaseCurrency = false): float
	{
		$price = $useBaseCurrency ? $this->priceInBaseCurrency : $this->price;

		return round($price / (1 + ($this->getVatRate() / 100)), 2);
	}

	public function getPrice(bool $useBaseCurrency = false): float
	{
		$price = $useBaseCurrency ? $this->priceInBaseCurrency : $this->price;
		foreach ($this->getChilds() as $child)
			$price += $child->getQuantity() * $child->getPrice($useBaseCurrency);

		return $price;
	}

	public function getPriceWithoutVat(bool $useBaseCurrency = false): float
	{
		return round($this->getPrice($useBaseCurrency) / (1 + ($this->getVatRate() / 100)), 2);
	}

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function getTotalPrice(bool $useBaseCurrency = false): float
	{
		$childPrice = 0;
		foreach ($this->getChilds() as $child)
			$childPrice += $child->getTotalPrice($useBaseCurrency);
		$total = $this->getQuantity() * ($this->getPriceWithoutChilds($useBaseCurrency) + $childPrice);

		return $total;
	}

	public function getTotalPriceWithoutVat(bool $useBaseCurrency = false): float
	{
		$childPrice = 0;
		foreach ($this->getChilds() as $child)
			$childPrice += $child->getTotalPriceWithoutVat($useBaseCurrency);
		$total = $this->getQuantity() * ($this->getPriceWithoutChildsWithoutVat($useBaseCurrency) + $childPrice);

		//		$total = round($this->getPriceWithoutVat($useBaseCurrency) * $this->getQuantity(), 2);
		//
		//		foreach ($this->getChilds() as $child)
		//			$total += $child->getTotalPriceWithoutVat($useBaseCurrency);

		return $total;
	}

	public function setData(array $arr): self
	{
		$this->data = $arr;

		return $this;
	}

	public function addData(string $key, $value): self
	{
		$this->data[$key] = $value;

		return $this;
	}

	public function getData(?string $key = null, $default = null)
	{
		if ($key) {
			return $this->data[$key] ?? $default;
		}

		return $this->data;
	}

	public function getVatRate(): int { return $this->vatRate; }

	public function setVatRate(int $vatRate): self
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	public function getUploadedFilesDir(): string { return UPLOADS_DIR . DS . 'eshoporders-cart-uploads' . DS . $this->id; }

	public function getUploadedFiles(): array
	{
		$files = [];

		foreach (glob($this->getUploadedFilesDir() . DS . '*') as $file) {
			$files[] = str_replace(WWW_DIR, '', $file);
		}

		return $files;
	}

	/****************
	 * == Gifts
	 */

	/** @return CartItemGift[] */
	public function getGifts(): array { return $this->gifts; }

	public function addGift(CartItemGift $gift): self
	{
		$this->gifts[] = $gift;

		return $this;
	}

	/****************
	 * == Childs
	 */

	public function addChild(CartItem $item)
	{
		$this->childs[$item->id] = $item;
		$item->parentId          = $this->id;
	}

	public function getChilds(): array { return $this->childs; }

	public function removeChild($k): void { unset($this->childs[$k]); }

	/**
	 * Grams
	 * @return int
	 */
	public function getTotalWeight(): int
	{
		return $this->weight ? $this->weight * $this->quantity : 0;
	}
}
