<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_orders__cart", indexes={@ORM\Index(name="ident", columns={"ident"})})
 * @ORM\Entity
 */
class Cart
{
	use TId;

	/**
	 * @ORM\Column(name="ident", type="string", nullable=true)
	 */
	public string $ident;

	/**
	 * @var CartItem[]
	 * @ORM\OneToMany(targetEntity="CartItem", mappedBy="cart", indexBy="ident")
	 */
	public $cartItems;

	/**
	 * @ORM\Column(name="order_gift_id", type="integer", nullable=true, options={"unsigned": true})
	 */
	public ?int $orderGiftId = null;

	/**
	 * @var DateTimeInterface
	 * @ORM\Column(name="created", type="datetime", nullable=false, options={"default":"CURRENT_TIMESTAMP"})
	 */
	private $created;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * @ORM\Column(name="last_activity", type="datetime", nullable=true, options={"default":"CURRENT_TIMESTAMP"})
	 */
	public ?DateTimeInterface $lastActivity = null;

	public function __construct(string $ident)
	{
		$this->ident        = $ident;
		$this->cartItems    = new ArrayCollection();
		$this->created      = new DateTime();
		$this->lastActivity = new DateTime();
		$this->moreData     = [];
	}

	/******
	 * === Cart Items
	 */

	/**
	 * @return ArrayCollection|CartItem[]
	 */
	public function getCartItems() { return $this->cartItems; }

	public function addCartItem($itemRaw)
	{
		$this->cartItems->add($itemRaw);
	}

	public function removeCartItem($itemId)
	{
		$this->cartItems->remove($itemId);
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

	public function getMoreData(): array { return $this->moreData ?? []; }

	public function getMoreDataValue(string $key)
	{
		return $this->moreData[$key] ?? null;
	}

	public function setMoreData(array $data): self
	{
		$this->moreData = $data;

		return $this;
	}

	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}
}

