<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Affiliate;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\AffiliateCampaigns;
use EshopOrders\AdminModule\Model\AffiliatePartners;
use EshopOrders\FrontModule\Model\EshopOrdersCache;
use EshopOrders\Model\Entities\Affiliate\AffiliateCampaign;
use Exception;
use InvalidArgumentException;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;

class CampaignForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	public ?AffiliateCampaign $affiliateCampaign = null;

	public function __construct(
		protected AffiliatePartners  $affiliatePartners,
		protected AffiliateCampaigns $affiliateCampaigns,
		protected EshopOrdersCache   $eshopOrdersCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->setAjax();

		$form->addText('ident', 'eshopOrders.affiliate.ident')
			->setDescription('eshopOrders.affiliate.identDescription');
		$form->addText('name', 'eshopOrders.affiliate.name')
			->setRequired();
		$form->addSelect('partner', 'eshopOrders.affiliate.partner', $this->affiliatePartners->getForSelectOption())
			->setRequired();
		$form->addText('commission', 'eshopOrders.affiliate.commission')
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setHtmlAttribute('max', 100)
			->setHtmlAttribute('step', 1)
			->setDefaultValue(2)
			->setRequired();

		$form->addSaveCancelControl('saveControl');

		if ($this->id) {
			$form->getComponent('ident')->setDisabled();
		}

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (!$this->id) {
			$ident  = Strings::webalize($values->ident ?: $values->name);
			$exists = $this->affiliateCampaigns->getEr()->findOneBy([
				'ident'   => $ident,
				'partner' => $values->partner,
			]);

			if ($exists) {
				$form->getComponent('ident')->addError('eshopOrders.affiliate.identExists');
			}
		}

		if ($form->getErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$ident = Strings::webalize($values->ident ?: $values->name);

			$partner = $this->affiliatePartners->get($values->partner);

			/** @var AffiliateCampaign $campaign */
			$campaign = $this->id ? $this->affiliateCampaigns->get($this->id) : new AffiliateCampaign($values->name, $ident, $partner);

			$campaign->name       = $values->name;
			$campaign->partner    = $partner;
			$campaign->commission = (int) $values->commission;

			$this->em->persist($campaign);
			$this->em->flush();

			$this->eshopOrdersCache->cleanAffiliate();

			return true;
		} catch (Exception $e) {
			bdump($e->getMessage());
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setCampaign(int $id): void
	{
		$this->id                = $id;
		$this->affiliateCampaign = $this->affiliateCampaigns->get($id);

		if (!$this->affiliateCampaign) {
			throw new InvalidArgumentException();
		}

		$d = [
			'ident'      => $this->affiliateCampaign->ident,
			'name'       => $this->affiliateCampaign->name,
			'partner'    => $this->affiliateCampaign->partner->getId(),
			'commission' => $this->affiliateCampaign->commission,
		];

		$this['form']->setDefaults($d);
	}
}
