<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\Model\Config;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\Entities\GroupCustomers;

class GroupsCustomersGrid extends BaseControl
{
	public function __construct(
		protected GroupsCustomers $groupsCustomersService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->groupsCustomersService->getEr()->createQueryBuilder('gc');
		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnLink('short', 'eshopOrders.defaultGrid.short', 'Customers:editGroup');
		$grid->addColumnText('name', 'eshopOrders.defaultGrid.name');

		if (Config::load('enablePriceLevels', false)) {
			$grid->addColumnText('productsSale', 'eshopOrders.groupCustomer.productsSale')
				->setRenderer(fn(GroupCustomers $row): string => $row->getProductsSale() !== 0.0 ? number_format($row->getProductsSale(), 2, ',', ' ') . ' %' : '');
		}

		$grid->addColumnStatus('isPublished', 'default.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Actions
		$grid->addAction('edit', '', 'Customers:editGroup')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->groupsCustomersService->remove($id)) {
			$this->presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		} else {
			$this->presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(string $id, string $newStatus): void
	{
		if ($this->groupsCustomersService->setPublish((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}
}
