<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Expedition;

use Core\Model\Event\CreateFormEvent;
use Core\Model\FileRequests;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders;
use EshopOrders\AdminModule\Model\OrderStatuses;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\Multiplier;
use Nette\Application\UI\Presenter;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class OrdersGrid extends BaseControl
{
	/** @var OrderExpedition[] */
	protected array $ordersForExpedition;

	protected ?array $cOrders = null;

	public array $onRedraw = [];

	public function __construct(
		protected Orders        $orders,
		protected LinkGenerator $linkGenerator,
		protected Statuses      $statusService,
		protected OrderStatuses $orderStatuses,
		protected FileRequests  $fileRequests
	)
	{
		$this->ordersForExpedition = $this->getOrders();
	}

	public function render(): void
	{
		$this->template->orders = $this->getOrders();
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentExpeditionForm(): Multiplier
	{
		return new Multiplier(function($arg) {
			[$orderId, $speditionIdent] = explode('__', $arg);
			$form = $this->createForm();
			$form->setAjax(false);
			$form->addHidden('orderId')
				->setDefaultValue($orderId)
				->setRequired();
			$form->addText('quantity', 'eshopOrders.expeditionOrdersGrid.form.quantity')
				->setHtmlType('number')
				->setPlaceholder($this->t('eshopOrders.expeditionOrdersGrid.form.quantity'))
				->addCondition($form::FILLED)
				->addRule($form::INTEGER);

			if ($speditionIdent !== '' && $speditionIdent !== '0') {
				$form->addCustomData('order', $this->ordersForExpedition[$orderId]);
				$event          = new CreateFormEvent($form, $this->getPresenter(false) instanceof Presenter ? $this->template : null);
				$event->control = $this;
				$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::createForm-' . $speditionIdent);
			}

			$form->onSuccess[] = $this->formSuccess(...);

			return $form;
		});
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		if (!Validators::isNumericInt($values->quantity) || empty($values->quantity)) {
			$values->quantity = 1;
		}

		$this->handleProcessOrder((int) $values->orderId, $values->quantity, $values->offsetExists('advancedOptions') ? (array) $values->advancedOptions : []);
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	public function handleRemove(int $orderId): void
	{
		if ($this->orderStatuses->removeProcessingStatus($orderId)) {
			$this->presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.removed');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
		$this->redrawControl('grid');
	}

	public function handleProcessOrder(int $orderId, int $quantity, array $advancedOptions = []): void
	{
		$presenter = $this->presenter;

		// Export
		$exportResult = $this->orders->sendExpeditionData([$orderId], $quantity, $advancedOptions);
		$exportErrors = $exportResult['errorCount'];
		$exportOks    = $exportErrors['okCount'];

		if (isset($exportResult['message']) && is_array($exportResult['message'])) {
			foreach ($exportResult['message'] as $messages) {
				if (!is_array($messages)) {
					continue;
				}
				foreach ($messages as $k => $m) {
					match ($k) {
						'danger' => $presenter->flashMessageDanger($m),
						'warning' => $presenter->flashMessageWarning($m),
						default => $presenter->flashMessageInfo($m),
					};
				}
			}
		}

		if ($exportErrors > 0) {
			$presenter->flashMessageDanger('eshopOrders.expeditionOrdersGrid.messages.exportError');
		} else if ($exportErrors === 0 && $exportOks > 0) {
			$presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.messages.exported');
		}

		// Label
		$labelResult = $this->orders->generateLabel([$orderId]);
		$labelFiles  = $labelResult['files'];

		if ($labelFiles) {
			$file = $labelFiles[0];

			if (str_starts_with((string) $file, 'http')) {
				$presenter->payload->fileRequests[] = [
					'name' => 'Label',
					'url'  => $this->linkGenerator->link('Core:Admin:FileResponse:', [base64_encode(serialize([
							'externalLink' => $file,
						]))]
					),
				];
				$presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.messages.labelGenerated');
			} else {
				$file = $this->fileRequests->add(base64_encode(serialize([
					'file'        => $file,
					'filename'    => strlen((string) $file) > 50 ? 'file' : basename((string) $file),
					'download'    => false,
					'contentType' => 'application/pdf',
				])));

				$arr                                = [
					'name' => 'Label',
					'url'  => $this->linkGenerator->link('Core:Admin:FileResponse:', [(string) $file->getId()]),
				];
				$presenter->payload->fileRequests[] = $arr;
				$presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.messages.labelGenerated');
			}

			if ($this->statusService->changeStatus([$orderId], EshopOrdersConfig::loadString('expeditionOrdersGrid.statusAfterExport') ?: 'spedition')) {
				$presenter->flashMessageSuccess('eshopOrders.order.statusChange');
			} else {
				$presenter->flashMessageDanger('eshopOrders.order.statusChangeFailed');
			}
		} else {
			$presenter->flashMessageDanger('eshopOrders.expeditionOrdersGrid.messages.labelGenerateError');
		}

		$this->cOrders = null;

		$presenter->redrawControl('flashes');
		$this->redrawControl('grid');

		foreach ($this->onRedraw as $v) {
			$v();
		}
	}

	protected function getOrders(): array
	{
		if ($this->cOrders === null) {
			$this->cOrders = $this->orders->getOrdersForExpedition();
		}

		return $this->cOrders;
	}
}
