<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\AdminModule\Components\Invoice\IQuickEditFormFactory;
use EshopOrders\AdminModule\Components\Invoice\QuickEditForm;
use EshopOrders\AdminModule\Components\OrderCurrency\IOrderCurrencyFormFactory;
use EshopOrders\AdminModule\Components\OrderCurrency\OrderCurrencyForm;
use EshopOrders\AdminModule\Components\OrderInvoice\IOrderInvoiceFormFactory;
use EshopOrders\AdminModule\Components\OrderInvoice\OrderInvoiceForm;
use EshopOrders\AdminModule\Model\OrderGifts;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Statuses;
use Exception;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;
use Users\Model\Users;

class OrderForm extends BaseControl
{
	public ?Order $order = null;

	public function __construct(
		protected Orders                                $ordersService,
		protected Speditions                            $speditionsService,
		protected Payments                              $paymentsService,
		protected CardsPaymentService                   $cardPaymentService,
		protected IOrderItemsGridFactory                $orderItemsGridFactory,
		protected IOrderStatusesGridFactory             $orderStatusesGridFactory,
		protected IIsPaidSwitchFactory                  $isPaidSwitchFactory,
		protected OrderGifts                            $orderGifts,
		protected Sellers                               $sellersService,
		protected PriceFilter                           $priceFilter,
		protected IEnableInvoiceGenerationSwitchFactory $enableInvoiceGenerationSwitchFactory,
		protected ICarrierHistoryGridFactory            $carrierHistoryGridFactory,
		protected Users                                 $usersService,
		protected Statuses                              $statusesService,
		protected IQuickEditFormFactory                 $editDUZPFormFactory,
		protected IOrderCurrencyFormFactory             $orderCurrencyFormFactory,
		protected IOrderInvoiceFormFactory              $orderInvoiceFormFactory,
		protected IExpeditionConfirmFactory             $expeditionConfirmFactory,
		protected IOrderSettingsFactory                 $orderSettingsFactory,
)
	{
	}

	public function render(): void
	{
		$this->template->paymentsService = $this->paymentsService;
		$this->template->order           = $this->order;
		$this->template->seller          = $this->sellersService->getSellerForSite($this->order->site->getIdent());
		$this->template->showPaid        = EshopOrdersConfig::load('ordersGrid.showCardHistory') && (
				EshopOrdersConfig::load('showPaidFieldForAllOrders', false) === true
				|| $this->order->getPaymentIdent() === 'card');

		if (EshopOrdersConfig::load('ordersGrid.showCardHistory') && $this->order->getPaymentIdent() == 'card') {
			$this->template->paymentHistory = $this->cardPaymentService->getHistoryOfOrder($this->order->getId());
		}

		$this->template->expeditionConfirmRulesOk = $this->order && $this->expeditionConfirmRulesOk($this->order);

		$this->template->render($this->getTemplateFile());
	}

	public function setOrder(int $id): void
	{
		$this->order = $this->ordersService->getFullOrder($id);

		if (!$this->order instanceof Order) {
			$this->presenter->error('default.notFound');
		}
	}

	public function redrawStatuses(): void
	{
		$this['orderStatusesGrid']['grid']->reload();
	}

	public function handleGiftReadyToDeliveryChange(string $id, string $newStatus): void
	{
		if ($this->orderGifts->setReadyToDelivery((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this->redrawControl('gifts');
		$this->redrawControl('expeditionForm');
		$this->presenter->redrawControl('flashes');
	}

	public function handleMessageReadyToDeliveryChange(string $id, string $newStatus): void
	{
		if ($this->ordersService->setMessageReadyToDelivery((int) $id, (bool) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this->redrawControl('message');
		$this->redrawControl('expeditionForm');
		$this->presenter->redrawControl('flashes');
	}

	public function handleRemoveGift(int $id): void
	{
		$orderGift = $this->orderGifts->get($id);
		$this->eventDispatcher->dispatch(new Event(['entity' => $orderGift]), 'eshopOrders.gifts.beforeDelete');

		$this->em->remove($orderGift);
		$this->em->flush();

		$this->presenter->flashMessageSuccess('default.removed');

		$this->redrawControl('gifts');
		$this->presenter->redrawControl('flashes');
	}

	public function handleValidate(string $type, string $val): void
	{
		if ($type === 'idNumber') {
			$this->ordersService->validateIdNumber($this->order, (int) $val);
			$this->presenter->flashMessageSuccess('default.saved');
		} else if ($type === 'vatNumber') {
			$this->ordersService->validateVatNumber($this->order, (int) $val);
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.order.validateNotFound');
		}

		$this->presenter->redirect('this');
	}

	public function handleChangePayment(int $id): void
	{
		$payment = $this->paymentsService->get($id);

		if (!$payment) {
			$this->presenter->flashMessageDanger('eshopOrders.paymentNotFound');
		} else {
			try {
				$this->paymentsService->setOrderPayment(
					$this->order,
					$payment,
					$payment->getName()
				);
				$this->presenter->flashMessageSuccess('eshopOrders.orderPaymentForm.edited');
			} catch (Exception) {
				$this->presenter->flashMessageDanger('default.error');
			}
		}

		$this->presenter->redrawControl('flashes');
		$this->redrawControl('payment');
		$this->redrawControl('spedition');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderItemsGrid(): OrderItemsGrid
	{
		return $this->orderItemsGridFactory->create($this->order);
	}

	protected function createComponentOrderStatusesGrid(): OrderStatusesGrid
	{
		$control = $this->orderStatusesGridFactory->create();
		if ($this->order instanceof Order) {
			$control->setOrder((int) $this->order->getId());
		}

		return $control;
	}

	protected function createComponentExpeditionConfirm(): ExpeditionConfirm
	{
		return $this->expeditionConfirmFactory->create();
	}

	public function createComponentExpeditionForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)->setAjax(false);
		$form->addTextArea('message', 'eshopOrders.default.statusMessage');
		$form->addBool('sendEmailToCustomer', 'eshopOrders.default.sendEmailToCustomer')
			->setDefaultValue(1);

		$event                = new CreateFormEvent($form, $this->presenter ? $this->template : null);
		$event->control       = $this;
		$event->data['order'] = $this->ordersService->get($this->order->getId());
		$this->eventDispatcher->dispatch($event, self::class . '::createExpeditionForm');

		if ($this->order && !$this->expeditionConfirmRulesOk($this->order)) {
			$form->addSubmit('save', 'eshopOrders.itemsGrid.dispatch')
				->getControlPrototype()->addHtml('<i class="fa fa-paper-plane"></i>');
		}

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->em->beginTransaction();

			try {
				$order = $this->ordersService->get($this->order->getId());
				if (!$order) {
					return;
				}

				$values->expeditionConfirmRulesOk = $this->expeditionConfirmRulesOk($order);
				$values->status                   = OrderStatus::STATUS_SPEDITION;
				$status                           = $this->statusesService->getReference(OrderStatus::STATUS_SPEDITION);
				$userRef                          = $this->usersService->getReference($this->presenter->getUser()->id);

				$orderStatus = new OrderStatus($this->ordersService->getReference((int) $order->getId()), $status, $userRef);
				$orderStatus->setMessage($values->message);

				$event                   = new FormSuccessEvent(
					$form,
					$values,
					$this->presenter ? $this->template : null,
					$this->presenter ?: null);
				$event->custom['entity'] = $orderStatus;
				$this->eventDispatcher->dispatch($event, OrderStatusForm::class . '::formSuccess');

				$this->em->persist($orderStatus);
				$this->em->flush();

				$event                  = new FormSuccessEvent($form, $values,
					$this->presenter ? $this->template : null, $this->presenter ?: null,
				);
				$event->control         = $this;
				$event->custom['order'] = $order;
				$this->eventDispatcher->dispatch($event, self::class . '::expeditionFormSuccess');

				$this->em->commit();

				$this->presenter->flashMessageSuccess('eshopOrders.statusForm.added');
			} catch (Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}
				Debugger::log($e);

				return;
			}

			if ($values->sendEmailToCustomer) {
				try {
					if ($this->statusesService->sendOrderStatusEmail($order, $orderStatus)) {
						$this->presenter->flashMessageSuccess('eshopOrders.statusForm.emailSent');
					}
				} catch (Exception $e) {
					Debugger::log($e->getMessage(), 'orderStatusError');
					$this->presenter->flashMessageDanger('eshopOrders.statusForm.emailError');

					return;
				}
			}

			$this->presenter->redirect(':EshopOrders:Admin:Default:');
		};

		return $form;
	}

	protected function createComponentCarrierHistoryGrid(): CarrierHistoryGrid { return $this->carrierHistoryGridFactory->create($this->order); }

	protected function createComponentInvoiceQuickEditForm(): QuickEditForm { return $this->editDUZPFormFactory->create($this->order->getInvoice()); }

	protected function createComponentCurrencyChangeForm(): OrderCurrencyForm { return $this->orderCurrencyFormFactory->create($this->order); }

	protected function createComponentOrderInvoiceForm(): OrderInvoiceForm { return $this->orderInvoiceFormFactory->create($this->order); }

	protected function createComponentIsPaidSwitch(): IsPaidSwitch { return $this->isPaidSwitchFactory->create($this->order); }

	protected function createComponentEnableInvoiceGenerationSwitch(): EnableInvoiceGenerationSwitch { return $this->enableInvoiceGenerationSwitchFactory->create($this->order); }

	protected function createComponentOrderSettings(): OrderSettings { return $this->orderSettingsFactory->create($this->order); }

	protected function createComponentInternalNoteForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		$form->addTextArea('note')
			->setDefaultValue($this->order->getParam(Order::paramInternalNote))
			->setHtmlAttribute('rows', $this->order->getMessage() ? 3 : 7);
		$form->addSubmit('save', 'default.save', 'btn-success btn-xs');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): bool {
			try {
				if (!EshopOrdersConfig::load('orderForm.allowInternalNote')) {
					throw new Exception('Not allowed');
				}

				if ($values->note) {
					$this->order->setParam(Order::paramInternalNote, $values->note);
				} else {
					$this->order->removeParam(Order::paramInternalNote);
				}

				$this->em->persist($this->order);
				$this->em->flush();

				$this->presenter->flashMessageSuccess('default.saved');
			} catch (Exception) {
				$this->presenter->flashMessageDanger('default.error');
			}

			$this->presenter->redrawControl('flashes');

			return false;
		};

		return $form;
	}

	protected function expeditionConfirmRulesOk(Order $order): bool
	{
		if (!EshopOrdersConfig::load('orderForm.expeditionConfirm.enable')) {
			return false;
		}

		$rules       = EshopOrdersConfig::loadArray('orderForm.expeditionConfirm.rules') ?: [];
		$customerOk  = $rules['excludeCustomerEmail'] === [] || ($order->getCustomer() && !in_array($order->getCustomer()->getUser()->email, $rules['excludeCustomerEmail'], true));
		$speditionOk = $rules['spedition'] === [] || (in_array($order->getSpeditionIdent(), $rules['spedition'], true));

		return $customerOk && $speditionOk;
	}

}
