<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\FormValidateEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderSpeditionHelper;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\UI\Form;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Override;

class OrderItemForm extends BaseControl
{
	#[Persistent]
	public ?int $orderItemId = null;

	public ?OrderItem      $orderItem      = null;
	public ?OrderItemTexts $orderItemTexts = null;
	public ?Order          $order          = null;

	public function __construct(
		protected Orders               $ordersService,
		protected OrderItems           $orderItemsService,
		protected OrderSpeditionHelper $orderSpeditionHelper,
	)
	{
	}

	public function render(): void
	{
		$this->template->item = $this->orderItem;
		$this->template->render($this->getTemplateFile());
	}

	#[Override]
	public function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		if ($this->orderItemId && !$this->orderItem) {
			$this->setOrderItem($this->orderItemId);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	public function handleDelete(string $itemId): void
	{
		$this->orderItemsService->removeOrderItem((int) $itemId);

		if (EshopOrdersConfig::load('recalculateSpeditionPriceByItemsPriceVatRate')) {
			$this->orderSpeditionHelper->checkSpeditionVatRateByItemsPrice($this->order);
		}

		$this->presenter->flashMessageSuccess('default.removed');
		$this->presenter->redrawControl('flashes');

		$this->presenter['orderForm']->redrawControl('items');
		$this->presenter['orderForm']->redrawControl('priceTotal');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addInteger('quantity', 'eshopOrders.default.quantity')->setRequired()->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('price', 'eshopOrders.default.pricePiece')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');

		if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
			$form->addText('vat', 'eshopOrders.default.vat')
				->setDescription('eshopOrders.orderItem.vatEditDescription')
				->setRequired();
		}

		if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
			$form->addTextArea('note', 'eshopOrders.orderItem.note')
				->setHtmlAttribute('rows', 3);
		}

		$form->addSaveCancelControl();
		$form->onValidate[] = function(BaseForm $form, ArrayHash $values): void {
			$event                      = new FormValidateEvent(
				$form,
				$values,
				$this->presenter ? $this->template : null,
				$this->presenter ?: null
			);
			$event->custom['orderItem'] = $this->orderItem;
			$event->custom['order']     = $this->order;
			if ($this->presenter) {
				$event->presenter = $this->presenter;
			}

			$this->eventDispatcher->dispatch($event, self::class . '::formValidate');

			if ($form->hasErrors()) {
				$this->redrawControl('form');
			}
		};
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->orderItem instanceof OrderItem) {
				$orderItem    = $this->orderItem;
				$flashMessage = 'eshopOrders.itemForm.edited';
			} else {
				/** @var Product|null $productRef */
				$productRef = $this->em->getReference(Product::class, $values->product_id);

				$orderItem    = new OrderItem($productRef, $this->order);
				$flashMessage = 'eshopOrders.itemForm.added';
			}

			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);

			if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
				$orderItem->setVatRate((int) $values->vat);
			}

			if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
				$orderItem->setMoreDataValue('note', $values->note);
			}

			$this->orderItemsService->saveOrderItem($orderItem);

			if (EshopOrdersConfig::load('recalculateSpeditionPriceByItemsPriceVatRate')) {
				$this->orderSpeditionHelper->checkSpeditionVatRateByItemsPrice($this->order);
			}

			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrder(int $orderId): void
	{
		$this->order = $this->ordersService->get($orderId);
	}

	public function setOrderItem(int $orderItemId): void
	{
		$this->orderItem      = $this->orderItemsService->get($orderItemId);
		$this->orderItemTexts = $this->orderItem->getOrderItemText('cs');
		$this->orderItemId    = $orderItemId;

		if ($this->orderItem) {
			$i = $this->orderItem;
			$d = [
				'quantity' => $i->getQuantity(),
				'price'    => $i->getPrice(),
				'note'     => $i->getMoreDataValue('note'),
			];

			if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
				$d['vat'] = $i->getVatRate();
			}

			$this['form']->setDefaults($d);
		}
	}
}
