<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\FormSuccessEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Statuses;
use Exception;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;
use Users\Model\Users;

class OrderStatusForm extends BaseControl
{
	public ?Order $order = null;

	public function __construct(
		protected Orders   $ordersService,
		protected Users    $usersService,
		protected Statuses $statusesService,
	)
	{
	}

	public function render(): void
	{
		$this->template->showExternalInvoiceHint = EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile')
			&& !$this->order->getInvoice()
			&& !$this->order->enableInvoiceGeneration;
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setHtmlAttribute('enable-external-invoice-hint', EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile', false));

		$statuses   = $this->statusesService->getAll();
		$statusList = [];
		foreach ($statuses as $status) {
			$id = $status->getId();

			if ($id !== 'created') {
				$statusList[$id] = $status->getName();
			}
		}

		//TODO defaultne nevybrany zadny status, a validace na to ze musi byt vybrany
		$form->addSelect('status', 'eshopOrders.default.status', $statusList);
		$form->addTextArea('message', 'eshopOrders.default.statusMessage');
		$form->addBool('sendEmailToCustomer', 'eshopOrders.default.sendEmailToCustomer')
			->setDefaultValue(1);

		$form->addSaveCancelControl();
		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$order   = $this->order;
			$status  = $this->statusesService->getReference($values->status);
			$userRef = $this->usersService->getReference($this->getPresenter()->getUser()->id);

			$orderStatus = new OrderStatus($this->ordersService->getReference((int) $order->getId()), $status, $userRef);
			$orderStatus->setMessage($values->message);

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->presenter ? $this->template : null,
				$this->presenter ?: null);
			$event->custom['entity'] = $orderStatus;
			$this->eventDispatcher->dispatch($event, OrderStatusForm::class . '::formSuccess');

			$this->em->persist($orderStatus);
			$this->em->flush();

			$this->presenter->flashMessageSuccess('eshopOrders.statusForm.added');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		if ($values->sendEmailToCustomer) {
			try {
				if ($this->statusesService->sendOrderStatusEmail($order, $orderStatus)) {
					$this->presenter->flashMessageSuccess('eshopOrders.statusForm.emailSent');
				}
			} catch (Exception $e) {
				Debugger::log($e->getMessage(), 'orderStatusError');
				$this->presenter->flashMessageDanger('eshopOrders.statusForm.emailError');
				$this->redrawControl('form');

				return false;
			}
		}

		return true;
	}

	public function setOrder(int $orderId): void
	{
		$this->order = $this->ordersService->get($orderId);
	}
}
