<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\OrderCurrency;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Exchange;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Invoices;
use Exception;
use Nette\Utils\ArrayHash;

class OrderCurrencyForm extends BaseControl
{
	protected ?array $cCurrencies = null;

	public function __construct(
		protected Order      $order,
		protected Currencies $currencies,
		protected Exchange   $exchange,
		protected Invoices   $invoices,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->setAjax();

		$form->addSelect('currency', 'eshopOrders.order.currency', $this->getCurrencies())
			->setDefaultValue($this->order->currency->code ?? $this->currencies->getDefaultCode())
			->setRequired();
		$form->addText('rate', 'eshopOrders.order.currencyRate')
			->setDefaultValue($this->order->currency->rate ?? '')
			->setHtmlAttribute('data-type', 'decimal')
			->setHtmlAttribute('data-decimals', 6);
		$form->addSubmit('save', 'default.save')
			->setHtmlAttribute('data-confirm', $this->t('default.reallyConfirm'));

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$currency = $this->order->currency;

			if ($values->currency === $this->currencies->getDefaultCode()) {
				if ($currency instanceof OrderCurrency) {
					$this->em->remove($currency);
					$this->order->currency = null;
				}
			} else {
				if (!$currency instanceof OrderCurrency) {
					$currency = new OrderCurrency($this->order, $values->currency);

					$this->order->currency = $currency;
				}

				/** @var Currency|null $currencyEntity */
				$currencyEntity = $this->currencies->getEr()->findOneBy(['code' => $values->currency, 'site' => [null, $this->order->site->getIdent()]]);

				if ($values->rate) {
					$rate = (float) $values->rate;
				} else {
					$rate = $this->exchange->changeByDateReverse(1, $this->order->getCreatedTime(), $this->currencies->getDefaultCode(), $values->currency);
				}

				$currency->rate = (string) Strings::formatEntityDecimal($rate);

				if ($currencyEntity) {
					$currency->decimals = $currencyEntity->decimals;
				}

				$this->em->persist($currency);
			}

			$this->em->flush();

			if ($this->order->getInvoice() instanceof Invoice) {
				$this->invoices->regenerateInvoice($this->order->getInvoice()->getId());
			}
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		$this->presenter->redirect('this');
	}

	protected function getCurrencies(): array
	{
		if ($this->cCurrencies === null) {
			$this->cCurrencies = [];

			foreach ($this->currencies->getEr()->findAll() as $row) {
				/** @var Currency $row */
				if (
					$row->isActive
					&& (!$row->site || $row->site->getIdent() === $this->order->site->getIdent())
				) {
					$this->cCurrencies[$row->getCode()] = $row->getCode();
				}
			}
		}

		return $this->cCurrencies;
	}
}
