<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\PaymentSpedition;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Payments;
use EshopOrders\Model\PaymentSpeditions;

class PaymentSpeditionsGrid extends BaseControl
{
	public function __construct(
		protected ?string                      $country,
		protected PaymentSpeditions            $paymentSpeditionsService,
		protected Payments                     $paymentsService,
		protected Speditions                   $speditionsService,
		protected Sites                        $sitesService,
		protected IPaymentSpeditionFormFactory $formFactory,
		protected Countries                    $countriesService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->paymentSpeditionsService->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.addPaymentSpedition');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.editPaymentSpedition');
		$this['form']->setPaymentSpedition($id);
		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([]);

		$qb = $this->paymentSpeditionsService->getEr()->createQueryBuilder('ps')
			->addSelect('p, s, sites')
			->leftJoin('ps.payment', 'p')
			->leftJoin('ps.spedition', 's')
			->leftJoin('ps.sites', 'sites')
			->addOrderBy('s.position', 'ASC')
			->addOrderBy('p.position', 'ASC');

		if ($this->country) {
			$qb->innerJoin('ps.countries', 'c', Join::WITH, 'c.id = :country')
				->setParameter('country', $this->country);
		} else {
			$qb->leftJoin('ps.countries', 'c')
				->andWhere('c.id IS NULL');
		}

		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name');
		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name');
		if (count($this->sitesService->getAll()) > 1) {
			$grid->addColumnText('sites', 'eshopOrders.paymentSpedition.sites')->setRenderer(fn(PaymentSpedition $row): string => $row->sites->count() ? implode(', ', $row->sites->getKeys()) : '');
		}

		$grid->addColumnStatus('isPublished', 'default.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Filter
		$grid->addFilterSelect('spedition', '', ['' => ''] + $this->speditionsService->getForSelectOption());
		$grid->addFilterSelect('payment', '', ['' => ''] + $this->paymentsService->getForSelectOption());
		$grid->addFilterSelect('isPublished', '', [
			'' => '',
			1  => 'default.yes',
			0  => 'default.no',
		])->setTranslateOptions();
		$grid->addFilterSelect('sites', '', [null => ''] + $this->sitesService->getOptionsForSelect())
			->setCondition(function(QueryBuilder $qb, $value): void {
				$qb->andWhere('sites.ident = :site')->setParameter('site', $value);
			});

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		return $grid;
	}

	protected function createComponentForm(): PaymentSpeditionForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setPaymentSpedition((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[] = function(): void {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function(): void {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(string $id, string $newStatus): void
	{
		if ($this->paymentSpeditionsService->setPublish((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}

}
