<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Spedition;

use Core\Model\Parameters;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\EshopOrdersConfig;

class SpeditionsGrid extends BaseControl
{
	public function __construct(
		protected ?string               $country,
		protected Speditions            $speditionsService,
		protected ISpeditionFormFactory $formFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->speditionsService->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopOrders.title.addSpedition');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-lg';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.editSpedition');
		$this['form']->setSpedition($id);
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-lg';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->speditionsService->getEr()->createQueryBuilder('s')
			->orderBy('s.position', 'ASC');

		if ($this->country) {
			$qb->andWhere('s.country = :country')
				->setParameter('country', $this->country);
		} else {
			$qb->andWhere('s.country IS NULL');
		}

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('speditionsGrid:gridSortableRow!');

		//Columns
		$grid->addColumnText('name', 'default.name');
		$grid->addColumnText('ident', 'eshopOrders.paymentForm.itemIdent');
		$grid->addColumnStatus('isPublished', 'default.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		$columnPrice = $grid->addColumnNumber('price', 'eshopOrders.default.priceSimple');
		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$grid->addColumnNumber('freeFrom', 'eshopOrders.default.freeFromSimple');
		}

		$grid->addColumnNumber('availableFrom', 'eshopOrders.default.availableFromSimple');
		$grid->addColumnNumber('availableTo', 'eshopOrders.default.availableToSimple');

		if (Parameters::load('currency.default', 'CZK') !== 'CZK') {
			$columnPrice->setFormat(2, ',', ' ');
		}

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		return $grid;
	}

	protected function createComponentForm(): SpeditionForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setSpedition((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[] = function(): void {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function(): void {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id') ?: null;
		$position  = $request->getPost('position') ?: null;

		if ($id !== null && $position !== null && $this->speditionsService->setPosition((int) $id, (int) $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$presenter->redrawControl('speditionsGrid');
		$presenter->redrawControl('paymentSpeditionsGrid');
	}

	public function gridPublishChange(string $id, string $newStatus): void
	{
		if ($this->speditionsService->setPublish((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}
}
