<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model\Statistics;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Doctrine\ORM\QueryBuilder;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use EshopOrders\Model\Statistics\BaseStatistics;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class Statistics
{
	protected ?array $cTodayStatusIds = null;

	protected ?array $cTodayCreated = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected BaseStatistics         $baseStatistics
	)
	{
	}

	public function getTodayStatusIds(string $status): array
	{
		if (!isset($this->cTodayStatusIds[$status])) {
			$this->cTodayStatusIds[$status] = [];

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, os.id')
				         ->where('os.status = :status')
				         ->andWhere('os.created >= :today')
				         ->setParameters(new ArrayCollection([new Parameter('status', $status), new Parameter('today', (new DateTime())->setTime(0, 0, 0))]))->getQuery()->getArrayResult() as $row) {
				$this->cTodayStatusIds[$status][$row['order']] = $row;
			}
		}

		return $this->cTodayStatusIds[$status];
	}

	public function getBaseQueryBuilder(): QueryBuilder
	{
		return $this->em->getRepository(Order::class)->createQueryBuilder('o', 'o.id')
			->addSelect('p, s, os, curr, od, pp, ss, site, inv, corDoc')
			->join('o.payment', 'p')
			->leftJoin('p.payment', 'pp')
			->join('o.spedition', 's')
			->leftJoin('s.spedition', 'ss')
			->innerJoin('o.site', 'site')
			->innerJoin('o.orderStatuses', 'os')
			->leftJoin('o.currency', 'curr')
			->leftJoin('o.orderDiscounts', 'od')
			->leftJoin('o.addressInvoice', 'inv')
			->leftJoin('o.orderForCorrectiveTaxDocument', 'corDoc');
	}

	/**
	 * @return Order[]
	 */
	public function getTodayCreated(): array
	{
		if ($this->cTodayCreated === null) {
			$this->cTodayCreated = [];

			foreach ($this->getBaseQueryBuilder()
				         ->andWhere('os.created >= :today AND os.status = :status')
				         ->setParameters(new ArrayCollection([new Parameter('status', OrderStatus::STATUS_CREATED), new Parameter('today', (new DateTime())->setTime(0, 0, 0))]))
				         ->groupBy('o.id')
				         ->getQuery()->getResult() as $order) {
				/** @var Order $order */
				$this->cTodayCreated[$order->getId()] = $order;
			}
		}

		return $this->cTodayCreated;
	}

	public function getTodayOrders(): array
	{
		$result = [
			'count' => 0,
			'price' => 0,
		];

		$canceled = $this->getTodayStatusIds(OrderStatus::STATUS_CANCELED);

		foreach ($this->getTodayCreated() as $order) {
			if (isset($canceled[$order->getId()])) {
				continue;
			}

			$result['count']++;
			$result['price'] += $order->getPriceItemsDiscount();
		}

		return $result;
	}

	public function getMonthlyFile(): string
	{
		return DATA_DIR . '/eshopOrders/stats/monthly.json';
	}

	public function getMonthly(bool $cached = true): array
	{
		$file = $this->getMonthlyFile();
		$date = DateTime::from((int) filemtime($file));

		if ($cached && file_exists($file) && $date->diff(new DateTime())->h <= 25) {
			$result = Json::decode((string) file_get_contents($file), Json::FORCE_ARRAY);
		} else {
			$result = $this->generateMonthly();
		}

		return $result;
	}

	public function generateMonthly(): array
	{
		set_time_limit(3600);
		ini_set('memory_limit', '1G');
		$file = DATA_DIR . '/eshopOrders/stats/monthly.json';
		FileSystem::createDir(dirname($file));

		$canceled = [];
		$data     = [];
		$result   = [];
		$from     = (new DateTime())->modify('-2 years');

		$created = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.status) as status, IDENTITY(os.order) as order, os.created')
			         ->where('os.status = :stat')
			         ->andWhere('os.created >= :from')
			         ->setParameter('from', $from)
			         ->setParameter('stat', OrderStatus::STATUS_CREATED)
			         ->getQuery()->getArrayResult() as $row) {
			$created[$row['order']] = $row['created'];
		}

		foreach (array_chunk(array_keys($created), 500) as $chunk) {
			foreach ($this->baseStatistics->getCancelled($chunk) as $id) {
				$canceled[$id] = $id;
			}

			// Sleva objednavky
			foreach ($this->baseStatistics->getSales($chunk, true) as $id => $sales) {
				$data[$id]['discounts'] = $sales;
			}

			// Produkty
			foreach ($this->baseStatistics->getOrderItems($chunk) as $id => $orderItems) {
				$data[$id]['items'] = $orderItems;
			}

			// platba
			foreach ($this->baseStatistics->getPayments($chunk) as $id => $items) {
				$data[$id]['payments'] = $items;
			}

			// doprava
			foreach ($this->baseStatistics->getSpeditions($chunk) as $id => $items) {
				$data[$id]['speditions'] = $items;
			}

			// Invoice addr
			foreach ($this->baseStatistics->getAddrInvoice($chunk) as $id => $items) {
				$data[$id]['addrInvoice'] = $items;
			}
		}

		foreach ($data as $id => $vals) {
			if (isset($canceled[$id])) {
				continue;
			}

			$key = isset($created[$id]) ? $created[$id]->format('Y-m') : null;
			if (!$key) {
				continue;
			}

			$price = 0;
			foreach ($vals['items'] ?? [] as $item) {
				$countryVatRate = null;

				if (array_key_exists('addrInvoice', $vals) && $vals['addrInvoice'] !== null) {
					$countryVatRate = OrderHelper::checkCountryVatRate(
						$item['vatRate'],
						$vals['addrInvoice']['country'] ?: 'CZ',
						(bool) $vals['addrInvoice']['validatedVatNumber'],
						$vals['addrInvoice']['idNumber'],
						$vals['addrInvoice']['vatNumber'],
					);
				}

				$priceCalculatorItem = new PriceCalculatorItem('', $item['price'], $item['quantity'], $item['vatRate'], 'CZK');

				$itemPrice = $countryVatRate === 0
					? $priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat()
					: $priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();
				if ($item['saleValue']) {
					$sale = OrderHelper::calculateSaleValue($item['saleType'], $item['saleValue'], $item['quantity'], $priceCalculatorItem->getPriceWithVat()->getAmount()->toFloat());

					if ($sale) {
						$saleCalculatorItem = new PriceCalculatorItem('', (float) $sale, 1, $item['vatRate'], 'CZK');

						$itemPrice = $countryVatRate === 0
							? $priceCalculatorItem->getTotalWithoutVat()->minus($saleCalculatorItem->getTotalWithoutVat())->getAmount()->toFloat()
							: $priceCalculatorItem->getTotalWithVat()->minus($saleCalculatorItem->getTotalWithVat())->getAmount()->toFloat();
					}
				}
				$price += $itemPrice;
			}

			foreach ($vals['payments'] ?? [] as $item) {
				$price += $item['price'];
			}

			foreach ($vals['speditions'] ?? [] as $item) {
				$price += $item['price'];
			}

			foreach ($vals['discounts'] ?? [] as $discount) {
				$price -= $discount;
			}


			if (!isset($result[$key])) {
				$result[$key]         = $this->monthlyCreateRow($created[$id]);
				$result[$key]['date'] = $result[$key]['date']->format('Y-m');
			}

			$result[$key]['orders']++;
			$result[$key]['ordersPrice'] += $price;

			unset($data[$id]);
		}

		krsort($result, SORT_STRING);

		FileSystem::delete($file);
		FileSystem::write($file, Json::encode($result));

		return $result;
	}

	protected function monthlyCreateRow(\DateTime $dateTime): array
	{
		return [
			'date'          => $dateTime,
			'orders'        => 0,
			'ordersPrice'   => 0,
			'canceled'      => 0,
			'canceledPrice' => 0,
		];
	}
}
