<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use Core\Model\Application\AppState;
use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\Site;
use Core\Model\Event\Event;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\PhoneInput;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Helpers\CustomerHelper;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Gift;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Helpers\OrderSpeditionHelper;
use EshopOrders\Model\Helpers\Psc;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\Model\Statuses;
use Exception;
use Nette\ComponentModel\IComponent;
use Nette\Forms\Controls\SubmitButton;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Nette\Utils\Floats;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Override;
use Tracy\Debugger;
use Tracy\ILogger;
use Users\Model\Entities\User;

class OrderForm extends BaseControl
{
	public const sessionNamespace = 'eshopOrdersOrderForm';

	public static array      $personalColumns = ['firstName', 'lastName', 'company', 'phone', 'email', 'country',
		'street', 'city', 'postal', 'idNumber', 'vatNumber'];
	protected SessionSection $sessionSection;

	/** @var Spedition[]|null */
	protected ?array $speditions = null;

	/** @var Payment[]|null */
	protected ?array $payments = null;

	/** @var Dao\PaymentSpedition[]|null */
	private ?array $paymentSpeditions = null;

	protected ?Cart  $cart                = null;
	protected ?array $paymentsSpeditions  = null;
	protected ?array $cCountries          = null;
	protected array  $orderPageConfig;
	protected bool   $allowNewsletterFlag = false;

	public function __construct(
		array                                $data,
		protected IOrderCartDetailFactory    $orderCartDetailFactory,
		protected IPaySpedSummaryFactory     $paySpedSummaryFactory,
		protected IOrderSummaryFactory       $orderSummaryFactory,
		protected Orders                     $ordersService,
		protected CartFacade                 $cartFacade,
		protected CartHelper                 $cartHelperService,
		protected Speditions                 $speditionsService,
		protected Payments                   $paymentsService,
		protected PaymentSpeditions          $paymentSpeditionsService,
		protected Customers                  $customersService,
		protected Statuses                   $statusesService,
		protected Countries                  $countriesService,
		protected Session                    $session,
		protected \Users\Model\Security\User $user,
		protected ProductsFacade             $products,
		protected Invoices                   $invoices,
		protected InvoiceConfigRepository    $invoiceConfigRepository,
		protected Sites                      $sitesService,
		protected PriceFilter                $priceFilter,
		protected Sellers                    $sellers,
		protected Currencies                 $currencies,
		protected Psc                        $pscHelper,
		protected AppState                   $appState,
		protected OrderSpeditionHelper       $orderSpeditionHelper
	)
	{
		$this->sessionSection  = $this->session->getSection(self::sessionNamespace);
		$this->orderPageConfig = (array) $data['orderPage'];

		$this->cart = $this->cartFacade->getCart();

		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			self::$personalColumns[] = 'idVatNumber';
		}

		if (EshopOrdersConfig::load('orderForm.allowNewsletterFlag')) {
			$this->allowNewsletterFlag();
		}
	}

	public function render(): void
	{
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		$customer  = $this->getCustomer();
		$cart      = $this->cartFacade->getCart();

		if ($customer && CustomerHelper::checkCustomerMultipleQuantitiesInCart($customer, $cart) > 0) {
			return;
		}

		$this->cart = $cart;
		$this->template->setFile($this->getTemplateFile());

		$this->template->visibleCountSpedition = $this->orderPageConfig['visibleCountSpedition'];
		$this->template->visibleCountPayment   = $this->orderPageConfig['visibleCountPayment'];

		$this->template->paymentSpeditions = $this->getPaymentSpeditions();
		$this->template->speditions        = $this->getSpeditions();
		$this->template->payments          = $this->getPayments();
		$this->template->countries         = $this->getCountries();
		$this->template->cart              = $this->cart;

		$form                     = $this['form'];
		$this->template->thisForm = $form;

		$this->template->termsAndConditionsNavId = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'TermsAndConditionsNavId', 0);
		$this->template->gdprNavId               = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'GdprNavId', 0);

		$paymentSpeditions                           = $this->getPaymentsSpeditionsCombinations();
		$this->template->paymentSpeditionsStructured = $paymentSpeditions['structured'];
		$this->template->paymentSpeditionsReversed   = $paymentSpeditions['reversed'];

		$this->template->orderData = $this->sessionSection->orderFormData;

		$this->template->render();
	}

	#[Override]
	public function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		$sessionData = $this->sessionSection->orderFormData;

		$user = $this->getPresenter()->getUser();
		if ($user->isLoggedIn()) {
			/** @var User $user */
			$user = $user->getIdentity();

			$customer = $this->customersService->getByUser($user);

			if ($customer instanceof Customer) {
				$deliAddr = $customer->getAddressDelivery();

				if ($customer && $deliAddr && is_null($sessionData['useAddrDeli']) && $deliAddr->getFirstName() && $deliAddr->getLastName()) {
					$sessionData['useAddrDeli']          = true;
					$this->sessionSection->orderFormData = $sessionData;
				}
			}
		}

		if ($sessionData) {
			$this->setOrderData($sessionData);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleSaveStep2(): void
	{
		$request    = $this->getPresenter()->getHttpRequest();
		$data       = $this->sessionSection->orderFormData;
		$speditions = $this->getSpeditions();

		/** @var Spedition|null $sessionSpedition */
		$sessionSpedition = $speditions[$data['speditionCountry']][$data['spedition']] ?? null;

		$data['speditionCountry'] = $request->getPost('speditionCountry');
		foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v) {
			$data[$k] = (int) $request->getPost($v)[$data['speditionCountry']];
		}

		/** @var Spedition|null $spedition */
		$spedition = $speditions[$data['speditionCountry']][$data['spedition']] ?? null;

		if ($spedition && $spedition->isPickup && $spedition->getIdent() === 'online') {
			$data['disableDeliveryAddress'] = true;
			$data['useAddrDeli']            = false;
		} else {
			unset($data['disableDeliveryAddress']);
			unset($data['useAddrDeli']);
		}

		// Pokud menim dopravu z "pickup" na "adresu", tak odskrtnout predvyplnene udaje doruceni
		if ($sessionSpedition && $sessionSpedition->isPickup && $spedition && !$spedition->isPickup) {
			$data['useAddrDeli'] = false;
			$this['form']->getComponent('useAddrDeli')->setValue(false);
		}

		if ($data['speditionCountry']) {
			$this->appState->setCountry($data['speditionCountry']);
		}

		/** @phpstan-ignore-next-line */
		if (!isset($data['country'])) {
			$this['form']->getComponent('country')->setValue($data['speditionCountry']);
		}

		/** @phpstan-ignore-next-line */
		if (!isset($data['country2'])) {
			$this['form']->getComponent('country2')->setValue($data['speditionCountry']);
		}

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep2');
		$this->sessionSection->orderFormData = $data;

		if ($data['disableDeliveryAddressSpedition'] !== null) {
			$this['form']->getComponent('country2')->setValue($data['speditionCountry']);
		}

		$currencyChange = null;
		if (isset($data['payment'])) {
			$payments = $this->getPayments();
			$payment  = $payments[$data['speditionCountry']][$data['payment']] ?? null;

			if ($payment && $payment->currency && $this->currencies->getCurrent()->getCode() != $payment->currency) {
				$curr = $this->currencies->getActive()[$payment->currency] ?? null;

				if ($curr) {
					$currencyChange = $curr;
				}
			}
		}

		if (!$currencyChange && isset($data['payment'], $data['spedition'])) {
			foreach ($this->getPaymentSpeditions() as $ps) {
				if ($this->currencies->getCurrent()->getCode() != $ps->currency
					&& array_key_exists((string) $data['speditionCountry'], $ps->getCountries())
					&& $ps->getSpedition()->getId() === $data['spedition'] && $ps->getPayment()->getId() === $data['payment']) {
					$curr = $this->currencies->getActive()[$ps->currency] ?? null;

					if ($curr) {
						$currencyChange = $curr;
					}
					break;
				}
			}
		}

		if ($currencyChange) {
			$this->currencies->setCurrentCurrency($currencyChange);
			$this->presenter->flashMessageSuccess($this->t('eshopOrdersFront.orderPage.currencyPaymentChangeTo', ['curr' => $currencyChange->getCode()]), 5000);
			$this->presenter->redirect('this');
		}

		$this->redrawControl('step3');
		$this->redrawControl('orderCartDetail');
		$this->redrawControl('orderSummary');
	}

	public function handleSaveStep3(): void
	{
		$request    = $this->getPresenter()->getHttpRequest();
		$data       = $this->sessionSection->orderFormData;
		$form       = $this['form'];
		$formValues = $form->getValues();

		$countries        = $this->countriesService->getAllNameColumn();
		$speditionCountry = $data['speditionCountry'];

		$customer = $this->getCustomer();

		// Kontrola existujiciho uzivatele podle ico a emailu
		if (!$customer instanceof Customer) {
			if (
				EshopOrdersConfig::load('orderForm.checkExistEmail')
				&& $this->customersService->checkExistEmail($formValues->email)
			) {
				$this['form']->addError('eshopOrdersFront.orderPage.emailExistLogInOrChange');
			}

			if (
				EshopOrdersConfig::load('orderForm.checkIcoExist')
				&& $this->customersService->checkExistIco($formValues->idNumber)
			) {
				$this['form']->addError('eshopOrdersFront.orderPage.icoExistLogInOrChange');
			}
		}

		$data['companyOrder'] = $formValues->companyOrder;
		foreach (self::$personalColumns as $k) {
			if (!$customer || !$customer->addressInvoice || !EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				$data[$k] = $request->getPost($k);
			}

			$data[$k . '2'] = $request->getPost($k . '2');
		}

		$data['deliveryAddressesList'] = $formValues->deliveryAddressesList;

		if ($customer && $customer->addressInvoice && EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
			$addrInvo = $customer->getAddressInvoice();

			if ($addrInvo instanceof CustomerAddress) {
				$data = array_merge($data, [
					'companyOrder' => true,
					'company'      => $addrInvo->getCompany(),
					'idNumber'     => $addrInvo->getIdNumber(),
					'vatNumber'    => $addrInvo->getVatNumber(),
				]);

				foreach ($addrInvo->toArray() as $k => $v) {
					if ($k === 'country') {
						if (!$v) {
							$data[$k]            = $speditionCountry;
							$data['countryText'] = $countries[$speditionCountry];
						} else {
							$data[$k]            = $v;
							$data['countryText'] = $v;
						}
					} else {
						$data[$k] = $v;
					}
				}
			}
		} else {
			$data['invoiceAddressesList'] = $formValues->invoiceAddressesList;
			$data['countryText']          = $countries[$request->getPost('country')] ?? null;
		}

		$data['country2Text'] = $countries[$request->getPost('country2')] ?? null;

		if (!$this->pscHelper->validatePostal((string) $data['postal'], (string) $data['country'])) {
			$form->getComponent('postal')->addError('eshopOrdersFront.orderPage.badPsc');
		}

		foreach (['useAddrDeli', 'speditionCountry'] as $k) {
			$data[$k] = $request->getPost($k);
		}

		if ($data['useAddrDeli'] === 'on' || $data['useAddrDeli'] === true) {
			$data['useAddrDeli'] = true;

			if ($data['disableDeliveryAddress'] !== true) {
				foreach (['firstName2', 'lastName2', 'phone2', 'street2', 'city2', 'postal2'] as $col) {
					if (!$data[$col]) {
						$form->getComponent($col)->addError('default.formMessages.filled');
					}
				}

				if ($data['email2'] && !Validators::isEmail((string) $data['email2'])) {
					$form->getComponent('email2')->addError('default.formMessages.email');
				}

				/** @var PhoneInput|null $phone2Input */
				$phone2Input = $form->getComponent('phone2', false);

				if ($phone2Input) {
					if (strlen((string) $data['phone2']) < 9) {
						$phone2Input->addError('default.formMessages.phone_min');
					} else if (!PhoneInput::validatePhone($phone2Input)) {
						$phone2Input->addError(PhoneInput::$phoneMessages);
					}
				}

				if (!$this->pscHelper->validatePostal((string) $data['postal2'], (string) $data['country2'])) {
					$form->getComponent('postal2')->addError('eshopOrdersFront.orderPage.badPsc');
				}

				if ($data['speditionCountry'] !== $data['country2']) {
					$form->getComponent('country2')->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
				}
			}
		} else {
			$data['useAddrDeli'] = false;
			if ($data['country'] != $speditionCountry) {
				$this['form']->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
			}
		}


		if ($this->isIcoRequired() && $formValues['companyOrder'] === false) {
			$this['form']->addError('eshopOrdersFront.orderPage.companyOrderIsRequired');
		}

		if (isset($formValues['message']) && $formValues['message']) {
			$data['message'] = $formValues->message;
		}

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep3');
		$this['form']->getComponent('validatePersonal')->validate();
		$this->sessionSection->orderFormData = $data;

		if ($this['form']->hasErrors()) {
			$fieldErrors = [];
			foreach (self::$personalColumns as $col) {
				foreach (['', '2'] as $num) {
					if (
						isset($form[$col . $num])
						&& method_exists($form[$col . $num], 'hasErrors')
						&& method_exists($form[$col . $num], 'getErrors')
						&& $form[$col . $num]->hasErrors()
					) {
						$fieldErrors[$col . $num] = $form[$col . $num]->getErrors();
					}
				}
			}

			$this->getPresenter()->payload->fieldErrors      = $fieldErrors;
			$this->getPresenter()->payload->orderFormErrors  = $this['form']->getErrors();
			$this->getPresenter()->payload->orderFormIsValid = false;
		}

		$this->cartFacade->getCartsService()->reloadCart();
		$this->redrawControl('step3');
		$this->redrawControl('orderSummary');
	}

	public function isIcoRequired(): bool
	{
		if (EshopOrdersConfig::load('orderForm.icoIsRequired', false)) {
			return true;
		}

		$customer = $this->getCustomer();

		if ($customer && $customer->getGroupCustomers()) {
			$requireIco = $customer->getGroupCustomers()->getParams()['requireIco'] ?? false;

			return EshopOrdersConfig::load('customerGroup.enableRequireIco') && $requireIco;
		}

		return false;
	}

	public function handleSetSpedition(int $spedition): void
	{
		$data = $this->sessionSection->orderFormData;

		$data['spedition']                   = $spedition;
		$this->sessionSection->orderFormData = $data;
		$this->cartFacade->getCart();
		$this->redrawControl('orderCartDetail');
	}

	public function handleSetPayment(int $payment): void
	{
		$data = $this->sessionSection->orderFormData;

		$data['payment']                     = $payment;
		$this->sessionSection->orderFormData = $data;
		$this->cartFacade->getCart();
		$this->redrawControl('orderCartDetail');
	}

	public function handleChangeSpeditionCountry(): void
	{
		$this->redrawControl('orderCartDetail');
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function allowNewsletterFlag(bool $allow = true): void { $this->allowNewsletterFlag = $allow; }

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$this->cartFacade->getCart();
		$countries = $this->countriesService->getAllNameColumn();

		$currency = $this->currencies->getCurrent();
		if (!$currency instanceof Currency) {
			throw new Exception('Currency not found');
		}

		$form->addHidden('currency', $currency->getCode());
		$form->addGroup('speditions');

		$appStateCountry    = $this->appState->getCountry();
		$speditionCountries = $this->getCountries();
		$defaultCountry     = isset($speditionCountries[$appStateCountry]) ? $appStateCountry : key($speditionCountries);

		$form->addSelect('speditionCountry', 'eshopOrders.orderPage.country', $this->getCountries())
			->setTranslator(null)
			->setDefaultValue($defaultCountry)
			->setHtmlAttribute('data-change-link', $this->link('changeSpeditionCountry!'))
			->setRequired('eshopOrdersFront.orderPage.requireSpeditionCountry');

		$speditionsContainer = $form->addContainer('speditions');
		$paymentsContainer   = $form->addContainer('payments');
		$paySpedCombinations = $this->getPaymentsSpeditionsCombinations();

		if ($this->presenter->user->isLoggedIn() && EshopOrdersConfig::load('allowAddressManagement', false)) {
			$customer = $this->customersService->getByUserId((int) $this->presenter->user->getId());
			if ($customer instanceof Customer) {
				$addresses    = [];
				$addressesArr = [];
				$icoData      = [
					'idNumber'  => null,
					'vatNumber' => null,
				];
				$addrToModify = [];

				/** @var CustomerAddress[] $addressesList */
				$addressesList = $customer->addressesAvailable->toArray();

				foreach ($addressesList as $addr) {
					if ($customer->addressInvoice && $customer->addressInvoice->getId() === $addr->getId()) {
						continue;
					}

					$addresses[$addr->getId()]    = $addr->getDetailString();
					$addressesArr[$addr->getId()] = $addr->toArray();

					if (EshopOrdersConfig::load('customer.disableModifyIco')) {
						foreach ($icoData as $colKey => $colVal) {
							if (!$colVal && $addressesArr[$addr->getId()][$colKey]) {
								$icoData[$colKey]             = $addressesArr[$addr->getId()][$colKey];
								$addrToModify[$addr->getId()] = $addr->getId();
							}
						}
					}
				}

				if (EshopOrdersConfig::load('customer.disableModifyIco') && $addrToModify !== []) {
					foreach ($addressesArr as $id => $addr) {
						foreach ($icoData as $colKey => $colVal) {
							if ($colVal && !$addr[$colKey]) {
								$addressesArr[$id][$colKey]    = $colVal;
								$addressesList[$id]->{$colKey} = $colVal;
							}
						}

						$this->em->persist($addressesList[$id]);
						$this->em->flush();
					}
				}

				$addressesArr = Json::encode($addressesArr);

				$form->addSelect('deliveryAddressesList', 'eshopOrdersFront.orderPage.selectDeliveryAddress', ['' => $this->t('eshopOrdersFront.orderPage.otherDeliveryAddress')] + $addresses)
					->setTranslator(null)
					->setHtmlAttribute('data-list', $addressesArr)
					->setHtmlAttribute('data-name-suffix', '2');

				if ($customer->getAddressDelivery() instanceof CustomerAddress) {
					$form->getComponent('deliveryAddressesList')->setDefaultValue($customer->getAddressDelivery()->getId());
				}

				if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					$form->addSelect('invoiceAddressesList', 'eshopOrdersFront.orderPage.selectInvoiceAddress', $addresses)
						->setHtmlAttribute('data-list', $addressesArr)
						->setHtmlAttribute('data-name-suffix', '');

					if ($customer->getAddressInvoice() instanceof CustomerAddress) {
						$form->getComponent('invoiceAddressesList')->setDefaultValue($customer->getAddressInvoice()->getId());
					}
				}
			}
		}

		foreach ($this->getSpeditions() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['structured'][$country][$row->getId()])) {
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
				}
			}

			$speditionsContainer->addRadioList($country, 'eshopOrders.orderPage.spedition', $list)
				->addConditionOn($form->getComponent('speditionCountry'), BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requireSpedition');
		}

		foreach ($this->getPayments() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['reversed'][$country][$row->getId()])) {
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
				}
			}

			$paymentsContainer->addRadioList($country, 'eshopOrders.orderPage.payment', $list)
				->addConditionOn($form->getComponent('speditionCountry'), BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requirePayment');
		}

		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form->getComponent('speditionCountry'),
				$form->getComponent('payments'),
				$form->getComponent('speditions'),
			])->onClick[] = $this->handleSaveStep2(...);

		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addPhone('phone', 'eshopOrders.orderPage.phone')
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')
			->setRequired();
		$form->addSelect('country', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);
		$form->addCheckbox('companyOrder', 'eshopOrdersForm.orderPage.companyOrder');
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')
			->addConditionOn($form->getComponent('companyOrder'), BaseForm::EQUAL, true)->setRequired();
		$form->addText('company', 'eshopOrders.orderPage.company')
			->addConditionOn($form->getComponent('companyOrder'), BaseForm::EQUAL, true)->setRequired();
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		if ($this->getPresenter()->getUser()->isLoggedIn() && EshopOrdersConfig::load('customer.disableModifyIco')) {
			$form->getComponent('idNumber')->setDisabled();
			$form->getComponent('vatNumber')->setDisabled();
		}

		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			$form->addText('idVatNumber', 'eshopOrders.orderPage.idVatNumber');
		}

		if ($this->isIcoRequired()) {
			$form->getComponent('companyOrder')->setDefaultValue(true)->setRequired();
		}

		if (EshopOrdersConfig::load('orderForm.allowLoadFromAres')) {
			$form->getComponent('idNumber')
				->setDescription('<a href="javascript:;" class="btn secondary-bg-color" data-order-form-ico-load="' . Parameters::loadString('api.urls.loadIco') . '" data-error-text="' . $this->t('eshopOrdersFront.orderPage.icoLoadError') . '">' . $this->t('eshopOrdersFront.orderPage.autoFillByIco') . '</a>');
		}

		if (EshopOrdersConfig::load('orderForm.autocompletePsc')) {
			$form->getComponent('postal')
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form->getComponent('postal')->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
		}

		//fakturacni adresa
		$form->addCheckbox('useAddrDeli', 'eshopOrders.orderPage.useAddrDeli');
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('firstName2', 'eshopOrders.orderPage.name');
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName');
		$form->addText('email2', 'eshopOrders.orderPage.email');
		$form->addPhone('phone2', 'eshopOrders.orderPage.phone');
		$form->addText('street2', 'eshopOrders.orderPage.street');
		$form->addText('city2', 'eshopOrders.orderPage.city');
		$form->addText('postal2', 'eshopOrders.orderPage.postal');

		$form->addSelect('country2', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);

		if ($firstNameMaxLength = EshopOrdersConfig::load('orderForm.fields.firstName.maxLength')) {
			$form->getComponent('firstName')->addRule($form::MAX_LENGTH, null, $firstNameMaxLength);
			$form->getComponent('firstName2')->addRule($form::MAX_LENGTH, null, $firstNameMaxLength);
		}

		if ($lastNameMaxLength = EshopOrdersConfig::load('orderForm.fields.lastName.maxLength')) {
			$form->getComponent('lastName')->addRule($form::MAX_LENGTH, null, $lastNameMaxLength);
			$form->getComponent('lastName2')->addRule($form::MAX_LENGTH, null, $lastNameMaxLength);
		}

		if ($streetMaxLength = EshopOrdersConfig::load('orderForm.fields.street.maxLength')) {
			$form->getComponent('street')->addRule($form::MAX_LENGTH, null, $streetMaxLength);
			$form->getComponent('street2')->addRule($form::MAX_LENGTH, null, $streetMaxLength);
		}

		if ($cityMaxLength = EshopOrdersConfig::load('orderForm.fields.city.maxLength')) {
			$form->getComponent('city')->addRule($form::MAX_LENGTH, null, $cityMaxLength);
			$form->getComponent('city2')->addRule($form::MAX_LENGTH, null, $cityMaxLength);
		}

		if (EshopOrdersConfig::load('orderForm.autocompletePsc')) {
			$form->getComponent('postal2')
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form->getComponent('postal2')->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
		}


		$form->getComponent('country')->addRule(function($item, BaseForm $args) {
			$data = $this->sessionSection->orderFormData;
			/** @var ArrayHash $values */
			$values = $args->getUntrustedValues();

			if ($values->useAddrDeli == false) {
				return true;
			} else if ($values->useAddrDeli == true) {
				return true;
			} else {
				return $values['speditionCountry'] === $item->getValue();
			}
		}, 'eshopOrdersFront.orderPage.badSpeditionCountry', $form);

		$validateFields = [];
		foreach (self::$personalColumns as $v) {
			$validateFields[] = $form->getComponent($v);

			if (!in_array($v, ['idNumber', 'vatNumber']) && $form->getComponent($v . '2', false)) {
				$validateFields[] = $form->getComponent($v . '2');
			}
		}

		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope($validateFields)
			->onClick[] = $this->handleSaveStep3(...);

		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$form->addCheckbox('agreedOne', 'eshopOrders.orderPage.agreedTerms')
				->setDefaultValue(1);
			if (EshopOrdersConfig::load('orderForm.requiredGdprOneCheckbox', false)) {
				$form->getComponent('agreedOne')->setRequired('eshopOrders.orderPage.agreedTermsRequired');
			}
		} else {
			$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')->setRequired('eshopOrders.orderPage.agreedTermsRequired');

			if (EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false)) {
				$form->getComponent('agreedTerms')->setRequired(false)
					->setDefaultValue(1);
			}

			if (EshopOrdersConfig::load('orderForm.agreedQuestioning', true)) {
				$form->addCheckbox('agreedQuestioning', 'eshopOrdersFront.orderPage.agreedQuestioning')
					->setDefaultValue(1);

				if (EshopOrdersConfig::load('orderForm.agreedQuestioningInvert', false)) {
					$form->getComponent('agreedQuestioning')->setCaption('eshopOrdersFront.orderPage.agreedQuestioningInvert')
						->setDefaultValue(0);
				}
			} else {
				$form->addHidden('agreedQuestioning', 0);
			}
		}

		if ($this->allowNewsletterFlag) {
			$form->addCheckbox('agreedNewsletter', 'eshopOrdersFront.orderPage.agreedNewsletter')
				->setDefaultValue(1);

			if (EshopOrdersConfig::load('orderForm.allowNewsletterFlagInvert', false)) {
				$form->getComponent('agreedNewsletter')->setCaption('eshopOrdersFront.orderPage.agreedNewsletterInvert')
					->setDefaultValue(0);
			}
		}

		$form->addSubmit('submit', 'eshopOrders.orderPage.send')
			->setHtmlAttribute('class', ['submitButton', 'eshopOrdersNext4'])
			->setHtmlId('orderFormSubmit');

		$customer = $this->getCustomer();

		if ($customer instanceof Customer) {
			$addrDeli = $customer->getAddressDelivery();
			$addrInvo = $customer->getAddressInvoice();

			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				$form->getComponent('companyOrder')->setDisabled();
				$form->getComponent('idNumber')->setDescription('');

				foreach (self::$personalColumns as $col) {
					$form->getComponent($col)->setDisabled();
				}
			}

			if ($addrInvo instanceof CustomerAddress) {
				$form->setDefaults($addrInvo->toArray());

				if ($addrInvo->getIdNumber()) {
					$d = [
						'companyOrder' => true,
						'company'      => $addrInvo->getCompany(),
						'idNumber'     => $addrInvo->getIdNumber(),
						'vatNumber'    => $addrInvo->getVatNumber(),
					];

					if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
						$d['idVatNumber'] = $addrInvo->idVatNumber;
					}

					$form->setDefaults($d);

					$this->template->companyFilled = true;
				}
			}

			if ($addrDeli instanceof CustomerAddress) {
				$def = [];
				foreach ($addrDeli->toArray() as $k => $v) {
					$def[$k . '2'] = $v;
				}

				$form->setDefaults($def);
			}
		}

		if ($this->isIcoRequired()) {
			$form->getComponent('companyOrder')->setValue(true);
			$form->getComponent('idNumber')->setRequired();

			$this->template->companyFilled = true;
		}

		$form->onValidate[] = $this->formOnValidate(...);
		$form->onSuccess[]  = $this->formOnSuccess(...);

		$orderFormEvent           = new OrderFormEvent($form);
		$orderFormEvent->template = $this->template;
		$this->eventDispatcher->dispatch($orderFormEvent, 'eshopOrders.createOrderForm');

		return $form;
	}

	public function setOrderData(array $orderData): void
	{
		$this->eventDispatcher->dispatch(new SetOrderFormDataEvent($orderData, $this['form']), 'eshopOrders.setOrderFormData');

		$customer = $this->getCustomer();

		$form = $this['form'];
		$d    = [
			'messageAlt'       => $orderData['messageAlt'],
			'message'          => $orderData['message'],
			'useAddrDeli'      => $orderData['useAddrDeli'],
			'agreedNewsletter' => $orderData['agreedNewsletter'],
		];

		foreach (['deliveryAddressesList', 'invoiceAddressesList'] as $v) {
			if ($orderData[$v]) {
				$d[$v] = $orderData[$v];
			}
		}

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$d['agreedOne'] = $orderData['agreedOne'];
		} else {
			if (EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false)) {
				$orderData['agreedTerms'] = 1;
			}

			$d['agreedTerms']       = $orderData['agreedTerms'];
			$d['agreedQuestioning'] = $orderData['agreedQuestioning'];
		}

		$spedCountry = $orderData['speditionCountry'];
		$countries   = $this->countriesService->getAllNameColumn();
		if ($spedCountry) {
			if (array_key_exists($spedCountry, $form->getComponent('speditionCountry')->getItems())) {
				$d['speditionCountry'] = $spedCountry;
			}

			foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v) {
				/** @phpstan-ignore-next-line */
				if ($orderData[$k] && isset($form[$v][$spedCountry]) && array_key_exists($orderData[$k], $form[$v][$spedCountry]->getItems())) {
					$d[$k][$spedCountry] = $orderData[$k];
				}
			}
		}

		foreach (self::$personalColumns as $k) {
			if (!$customer || !$customer->getAddressInvoice() || !EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				if ($orderData[$k]) {
					$d[$k] = $orderData[$k];
				}
			} else {
				foreach ($customer->getAddressInvoice()->toArray() as $ak => $av) {
					if ($ak === 'country') {
						if (!$av) {
							$d[$k]            = $spedCountry;
							$d['countryText'] = $countries[$spedCountry];
						} else {
							$d[$ak]           = $av;
							$d['countryText'] = $av;
						}
					} else {
						$d[$ak] = $av;
					}
				}
			}

			if (!in_array($k, ['idNumber', 'vatNumber']) && $orderData[$k . '2']) {
				$d[$k . '2'] = $orderData[$k . '2'];
			}
		}

		$form->setDefaults($d);
	}

	public function formOnValidate(BaseForm $form): bool
	{
		/** @var ArrayHash $values */
		$values                              = $form->getUntrustedValues();
		$cart                                = $this->cartFacade->getCart();
		$cartItems                           = $cart->getCartItems();
		$this->template->formSummaryMessages = [];

		if ($cartItems === [] && EshopOrdersConfig::load('orderForm.enableEmptyCartValidation', true)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCartData'];
		}

		$spedCountry = $values->speditionCountry;
		$spedition   = $values->speditions[$spedCountry] ?? null;
		$payment     = $values->payments[$spedCountry] ?? null;

		$paymentsSpeditions = $this->getPaymentsSpeditionsCombinations();
		$paySped            = $paymentsSpeditions['structured'][$spedCountry][$spedition] ?? null;
		if (!$paySped || !in_array($payment, $paySped)) {
			$form->addError('eshopOrdersFront.orderPage.invalidPaymentSpeditionCombination');
			Debugger::log('payment: ' . $payment . ', spedition: ' . $spedition . ', coutry: ' . $spedCountry, 'paymentSpeditionCombination');
		}

		$remainingValue = $cart->getRemainingValueToCompleteOrder();
		if ($remainingValue > 0) {
			$form->addError(Html::el()
				->setHtml($this->t('eshopOrdersFront.orderPage.youNeedMoreToCompleteOrder', ['price' => $this->priceFilter->format($remainingValue)])),
				false,
			);
		}

		if (Floats::isLessThan($cart->getPriceTotal(), 0.0)) {
			$form->addError('eshopOrdersFront.orderPage.invalidOrderPrice');
		}

		$this->eventDispatcher->dispatch(new OrderFormEvent($form), 'eshopOrders.orderFormValidate');

		if ($form->getErrors() !== []) {
			Debugger::log($form->getErrors(), 'orderFormValidate');
		}

		if ($form->hasErrors()) {
			$errs = [];
			foreach ($form->getComponents(true) as $c) {
				if ($c instanceof BaseContainer || !method_exists($c, 'getErrors')) {
					continue;
				}

				if ($c->getErrors() && method_exists($c, 'getHtmlId')) {
					$errs[$c->getHtmlId()] = $c->getErrors();
				}
			}

			$this->getPresenter()->payload->orderFormErrors  = $errs;
			$this->getPresenter()->payload->orderFormIsValid = false;

			foreach ($form->getErrors() as $e) {
				$this->template->formSummaryMessages[] = $e;
			}

			$this->redrawControl('formMessages');

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');

			return false;
		} else {
			$this->getPresenter()->payload->orderFormIsValid = true;

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');
		}

		return true;
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		/** @var SubmitButton|null $submitControl */
		$submitControl = $form->getComponent('submit', false);
		if (!$submitControl || !$submitControl->isSubmittedBy()) {
			return false;
		}

		try {
			$orderFormData = $this->sessionSection->orderFormData;

			/** @var Site $currentSite */
			$currentSite = $this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent());
			$order       = new Order($currentSite);
			$order->lang = $this->translator->getLocale();

			$agreedQuestioning = $values->agreedQuestioning;
			$agreedNewsletter  = $values->agreedNewsletter;

			if (EshopOrdersConfig::load('orderForm.agreedQuestioningInvert', false)) {
				$agreedQuestioning = !$agreedQuestioning;
			}

			if (EshopOrdersConfig::load('orderForm.allowNewsletterFlagInvert', false)) {
				$agreedNewsletter = !$agreedNewsletter;
			}

			if ($values->speditionCountry && isset(EshopOrdersConfig::load('order.countryLangs')[$values->speditionCountry])) {
				$order->lang = EshopOrdersConfig::load('order.countryLangs')[$values->speditionCountry];
			}

			$order->setMessage($values->message ?: $values->messageAlt);
			$order->setAgreedTerms(
				(bool) $values->agreedTerms
				|| $values->agreedOne
				|| EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false)
			);

			if ($agreedQuestioning
				|| ($values->agreedOne && EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false))
			) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$this->em->persist($orderFlagQuestioning);
				$order->addFlag($orderFlagQuestioning);
			}

			if ($agreedNewsletter) {
				$orderFlagNewsletter = new OrderFlag(OrderFlag::TYPE_NEWSLETTER, true, $order);
				$this->em->persist($orderFlagNewsletter);
				$order->addFlag($orderFlagNewsletter);
			}

			$customer = $this->getCustomer();
			if ($customer && $agreedNewsletter) {
				$customer->user->newsletterSubscribed = (int) $agreedNewsletter;

				$this->em->persist($customer->user);
			}

			if ($customer && $customer->getAddressInvoice() && EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				$tmp     = $customer->getAddressInvoice();
				$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
				$addrInv->setFirstName($tmp->firstName);
				$addrInv->setLastName($tmp->lastName);
				$addrInv->setEmail($tmp->email);
				$addrInv->setPhone($tmp->phone);
				$addrInv->setStreet($tmp->street);
				$addrInv->setCity($tmp->city);
				$addrInv->setPostal($tmp->postal);
				if ($tmp->getCountry() instanceof Country) {
					$addrInv->setCountry($tmp->getCountry());
				} else {
					$addrInv->setCountry($this->countriesService->getReference($orderFormData['country']));
				}

				if ($tmp->company) {
					$addrInv->setCompany($tmp->company);
				}
				if ($tmp->idNumber) {
					$addrInv->setIdNumber($tmp->idNumber);
				}
				if ($tmp->vatNumber) {
					$addrInv->setVatNumber($tmp->vatNumber);
				}
				$addrInv->idVatNumber = $tmp->idVatNumber ?: null;
			} else {
				$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
				$addrInv->setFirstName($values->firstName);
				$addrInv->setLastName($values->lastName);
				$addrInv->setEmail($values->email);
				$addrInv->setPhone($values->phone);
				$addrInv->setStreet($values->street);
				$addrInv->setCity($values->city);
				$addrInv->setPostal($values->postal);
				if ($values->country) {
					$addrInv->setCountry($this->countriesService->getReference($values->country));
				}

				if ($values->company) {
					$addrInv->setCompany($values->company);
				}
				if ($values->idNumber) {
					$addrInv->setIdNumber($values->idNumber);
				}
				if ($values->vatNumber) {
					$addrInv->setVatNumber($values->vatNumber);
				}
				$addrInv->idVatNumber = $values->idVatNumber ?: null;
			}

			$this->em->persist($addrInv);

			$order->setAddressInvoice($addrInv);

			$payment   = $this->paymentsService->get((int) $values->payments[$values->speditionCountry]);
			$spedition = $this->speditionsService->get((int) $values->speditions[$values->speditionCountry]);

			if ($payment && $spedition && $invGenPaySpedComb = EshopOrdersConfig::loadArray('invoice.disableInvoiceGenerationAsDefaultOnPaySpedCombination') ?: []) {
				foreach ($invGenPaySpedComb as $combination) {
					if ($payment->ident === $combination['payment'] && $spedition->ident === $combination['spedition']) {
						$order->enableInvoiceGeneration = 0;
						break;
					}
				}
			}

			$addrDeli = null;
			if ($values->useAddrDeli) {
				$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($values->firstName2 ?: $addrInv->getFirstName());
				$addrDeli->setLastName($values->lastName2 ?: $addrInv->getLastName());
				$addrDeli->setEmail($values->email2 ?: $addrInv->getEmail());
				$addrDeli->setPhone($values->phone2 ?: $addrInv->getPhone());
				$addrDeli->setStreet($values->street2);
				$addrDeli->setCity($values->city2);
				$addrDeli->setPostal($values->postal2);
				if ($values->country2) {
					$addrDeli->setCountry($this->countriesService->getReference($values->country2));
				}
				$addrDeli->setCompany($values->company2);
				$this->em->persist($addrDeli);

				$order->setAddressDelivery($addrDeli);
			}

			$customer = $this->getOrCreateCustomer(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName(),
				$order->getAddressInvoice()->getLastName(),
				$order->getAddressInvoice()->getPhone());

			/** @phpstan-ignore-next-line */
			if ($customer) {
				$order->setCustomer($customer);

				if (EshopOrdersConfig::load('allowAddressManagement', false)) {
					$customerDeli = $values->deliveryAddressesList ? $customer->addressesAvailable->get($values->deliveryAddressesList) : null;
					$customerInvo = EshopOrdersConfig::load('customer.disableModifyInvoiceAddress') && $customer->addressInvoice
						? $customer->addressInvoice
						: $customer->addressesAvailable->get($values->invoiceAddressesList);
				} else {
					$customerDeli = $customer->getAddressDelivery();
					$customerInvo = $customer->getAddressInvoice();
				}

				if (!$customerInvo) {
					$customerInvo = new CustomerAddress($customer);
				}

				if (!$customerInvo->firstName || !EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					$customerInvo->fillFromOrderAddress($addrInv);
					$customer->setAddressInvoice($customerInvo);
					$this->em->persist($customerInvo);
				}

				if ($addrDeli && !$spedition->isPickup) {
					if (!$customerDeli) {
						$customerDeli = new CustomerAddress($customer);
					}

					$customerDeli->fillFromOrderAddress($addrDeli);
					$customer->setAddressDelivery($customerDeli);
					$this->em->persist($customerDeli);
				}

				if (EshopOrdersConfig::load('order.autoValidateIfUserIsValid')) {
					if (EshopOrdersConfig::load('order.allowValidateIdNumber')) {
						if ($addrInv && $addrInv->getIdNumber()) {
							$addrInv->validatedIdNumber = $customer->getAddressInvoice()->validatedIdNumber;
						}
						if ($addrDeli && $addrDeli->getIdNumber()) {
							$addrDeli->validatedIdNumber = $customer->getAddressDelivery()->validatedIdNumber;
						}
					}
					if (EshopOrdersConfig::load('order.allowValidateVatNumber')) {
						if ($addrInv && $addrInv->getVatNumber()) {
							$addrInv->validatedVatNumber = $customer->getAddressInvoice()->validatedVatNumber;
						}
						if ($addrDeli && $addrDeli->getVatNumber()) {
							$addrDeli->validatedVatNumber = $customer->getAddressDelivery()->validatedVatNumber;
						}
					}

					$this->em->persist($addrInv);

					if ($addrDeli instanceof OrderAddress) {
						$this->em->persist($addrDeli);
					}
				}
			}

			$cart       = $this->cartFacade->getCart();
			$cartItems  = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);

			/** @var \EshopOrders\Model\Entities\Spedition|null $speditionRef */
			$speditionRef = $this->speditionsService->getReference((int) $values->speditions[$values->speditionCountry]);

			$orderSpedition        = new OrderSpedition($speditionRef, $order);
			$orderSpeditionVatRate = OrderHelper::checkCountryVatRate(
				(int) $cart->spedition->vat,
				$addrInv->getCountry()->getId(),
				false,
				$addrInv->getIdNumber(),
				$addrInv->getVatNumber(),
			);
			$orderSpedition->setVatRate($orderSpeditionVatRate);
			$orderSpedition->setPrice($spedition->getPriceActual($cart, true));
			$order->setSpedition($orderSpedition);

			/** @var Payment|null $paymentRef */
			$paymentRef = $this->paymentsService->getReference((int) $values->payments[$values->speditionCountry]);

			$orderPayment = new OrderPayment($paymentRef, $order);
			$orderPayment->setVatRate($orderSpeditionVatRate);
			$orderPayment->setPrice($payment->getPriceActual($cart, true));
			$order->setPayment($orderPayment);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$order->getOrderStatuses()->add($orderActionCreated);

			$cartOrderGift = $cart->getOrderGift();
			if ($cartOrderGift instanceof Gift) {
				/** @var Product $productRef */
				$productRef = $this->em->getReference(Product::class, $cartOrderGift->productId);

				$orderGift        = new OrderGift($order, $productRef, $cartOrderGift->name);
				$orderGift->ean   = $cartOrderGift->ean;
				$orderGift->code1 = $cartOrderGift->code1;

				$this->em->persist($orderGift);
			}

			//			foreach ($cart->getDiscounts() as $k => $v) {
			//				$discount = new OrderDiscount($v->id, $order);
			//				$discount->setValue($v->amount);
			//				$discount->setType($v->type);
			//				$discount->setPrice($cart->calculateDiscount($v));
			//				$discount->setName($this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
			//				$this->em->persist($discount);
			//				$order->addOrderDiscount($discount);
			//			}

			$order->reloadPriceCalculator();

			$this->em->getConnection()->beginTransaction();
			try {
				$eventOrder                 = new OrderEvent($order);
				$eventOrder->addrDelivery   = $addrDeli;
				$eventOrder->addrInvoice    = $addrInv;
				$eventOrder->orderSpedition = $orderSpedition;
				$eventOrder->orderPayment   = $orderPayment;
				$eventOrder->orderItems     = $orderItems;
				$eventOrder->formData       = $values;
				$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');

				//ulozit objednavku
				$this->em->persist($order);

				//ulozit k ni adresy
				$this->em->persist($addrInv);

				if ($addrDeli instanceof OrderAddress) {
					$this->em->persist($addrDeli);
				}

				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);

				//ulozit k ni polozky
				foreach ($orderItems as $oi) {
					$this->em->persist($oi);
				}

				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);

				//pripadne nastavene priznaky
				if (isset($orderFlagQuestioning)) {
					$this->em->persist($orderFlagQuestioning);
				}

				// vytvoreni faktury
				if ($order->getInvoice() instanceof Invoice) {
					$this->em->persist($order->getInvoice());
				}

				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();

				$this->ordersService->moveCartFilesToOrder($cartItems, $orderItems);
				$this->em->getConnection()->commit();
				$order->reloadPriceCalculator();

				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				Debugger::log($e->getMessage(), ILogger::ERROR);
				Debugger::log($e, ILogger::ERROR);
				throw $e;
			}

			//dale uz pracujeme s objednavkou nactenou z DB
			/** @var Order $orderDetail */
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);

			if (EshopOrdersConfig::load('recalculateSpeditionPriceByItemsPriceVatRate')) {
				$this->orderSpeditionHelper->checkSpeditionVatRateByItemsPrice($orderDetail);
			}

			$this->cartFacade->getCartsService()->deleteCurrentCart();
			$this->sessionSection->orderFormData = [];

			// Spuštění události po dokončení objednávky jako odeslání emailu, zboží,...
			$this->eventDispatcher->dispatch(new OrderEvent($orderDetail), 'eshopOrders.orderOnSuccess');
			// Vytáhnutí zpráv z událostí
			$messages = $this->session->getSection('eshopOrders/orderSuccessMessages');

			if ($messages->success) {
				foreach ($messages->success as $v) {
					$this->presenter->flashMessageSuccess($v);
				}
			}

			if ($messages->error) {
				foreach ($messages->error as $v) {
					$this->presenter->flashMessageDanger($v);
				}
			}

			$messages->remove();
		} catch (Exception $e) {
			Debugger::log($e, 'exception');
			$form->addError($e->getMessage());

			return false;
		}

		$this->presenter->redirect(':EshopOrders:Front:Finished:orderFinished', ['orderIdent' => $orderDetail->getIdent()]);
	}

	protected function getOrCreateCustomer(?string $email, ?string $firstName = '', ?string $lastName = '', ?string $phone = ''): Customer
	{
		$userId = null;
		if ($this->presenter->user->isLoggedIn()) {
			$userId = (int) $this->presenter->user->getId();
		}

		return $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderCartDetail(): OrderCartDetail { return $this->orderCartDetailFactory->create(); }

	protected function createComponentPaySpedSummary(): PaySpedSummary
	{
		$control = $this->paySpedSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters((int) $orderData['spedition'], (int) $orderData['payment'], (float) $orderData['priceTotal']);

		return $control;
	}

	protected function createComponentOrderSummary(): OrderSummary
	{
		$control = $this->orderSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData ?: [];
		$control->setParameters($orderData);

		return $control;
	}

	/**
	 * @return Spedition[][]
	 */
	public function getSpeditions(): array
	{
		if ($this->speditions === null) {
			/** @var Spedition[] $raw */
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getSpedition()->getId()] = $ps->getSpedition()->getId();
				}

				$raw[$ps->getSpedition()->getId()] = $ps->getSpedition();
			}
			$raw = $this->speditionsService->filterByCart($raw, $this->cart);

			$this->speditions = $this->processPaySpedData($byCountry, $raw);

			$this->eventDispatcher->dispatch(new Event(['speditions' => &$this->speditions]), OrderForm::class . '::loadSpeditions');
		}

		return $this->speditions;
	}

	public function getPaymentsSpeditionsCombinations(): array
	{
		if ($this->paymentsSpeditions === null) {
			$filteredSpedition = $this->getSpeditions();
			$filteredPayments  = $this->getPayments();

			$paymentSpeditions        = $this->getPaymentSpeditions();
			$this->paymentsSpeditions = [
				'structured' => [],
				'reversed'   => [],
			];

			foreach ($paymentSpeditions as $item) {
				foreach ($item->getCountries() as $country) {
					/** @var Country $country */
					$spedId    = $item->getSpedition()->getId();
					$payId     = $item->getPayment()->getId();
					$countryId = $country->getId();

					if (!isset($filteredPayments[$countryId][$payId]) || !isset($filteredSpedition[$countryId][$spedId])) {
						continue;
					}

					$this->paymentsSpeditions['structured'][$countryId][$spedId][] = $payId;
					$this->paymentsSpeditions['reversed'][$countryId][$payId][]    = $spedId;
				}
			}
		}

		return $this->paymentsSpeditions;
	}

	/**
	 * @return Payment[][]
	 */
	public function getPayments(): array
	{
		if ($this->payments === null) {
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getPayment()->getId()] = $ps->getPayment()->getId();
				}

				$raw[$ps->getPayment()->getId()] = $ps->getPayment();
			}
			$raw = $this->paymentsService->filterByCartValue($raw, $this->cart->getCartItemsPrice());

			$this->payments = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->payments;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getPaymentSpeditions(): array
	{
		if ($this->paymentSpeditions === null) {
			$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();
		}

		return $this->paymentSpeditions;
	}

	public function getCountries(): array
	{
		if ($this->cCountries === null) {
			$this->cCountries = [];

			foreach ($this->getPaymentSpeditions() as $row) {
				foreach ($row->getCountries() as $country) {
					$this->cCountries[$country->getId()] = $country->getName();
				}
			}
		}

		return $this->cCountries;
	}

	protected function processPaySpedData(array $byCountry, array $raw): array
	{
		$result = [];

		foreach ($byCountry as $country => $rows) {
			foreach ($rows as $row) {
				if (!isset($raw[$row])) {
					continue;
				}

				$raw[$row]->country     = $country;
				$result[$country][$row] = $raw[$row];
			}
		}

		return $result;
	}

	protected function getCustomer(): ?Customer
	{
		/** @var User|null $user */
		$user = $this->user->getIdentity();

		/** @var Customer|null $customer */
		$customer = $user ? $this->customersService->getByUser($user) : null;

		return $customer;
	}
}
