<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\GroupCustomers as GroupCustomerEntity;
use EshopOrders\Model\Helpers\CustomersCache;
use EshopOrders\Model\Helpers\CustomersCacheHelper;
use Nette\Caching\Cache;

/**
 * @method GroupCustomerEntity|object|null getReference($id)
 * @method GroupCustomerEntity[]|null getAll()
 * @method GroupCustomerEntity|null get($id)
 */
class GroupCustomers extends BaseEntityService
{
	public const CUSTOMER = 'customer';

	protected $entityClass = GroupCustomerEntity::class;

	protected array $cGroups        = [];
	protected array $cGroupsByUsers = [];

	public function __construct(
		protected CustomersCache $customersCache,
	)
	{
	}

	public function getGroupArray(int $groupId): ?array
	{
		if (!array_key_exists($groupId, $this->cGroups)) {
			$this->cGroups[$groupId] = $this->customersCache->getCache()->load(CustomersCacheHelper::getCustomerGroupKey($groupId), function(&$dep) use ($groupId) {
				$dep = [
					Cache::EXPIRATION => '1 week',
					Cache::Tags       => ['groupCustomer'],
				];

				$result = $this->getEr()->createQueryBuilder('g')
					->addSelect('IDENTITY(g.inheritPricesGroup) as inheritPricesGroup')
					->addSelect('IDENTITY(g.useProductsSaleOnGroup) as useProductsSaleOnGroup')
					->where('g.id = :groupId')
					->setParameter('groupId', $groupId)
					->setMaxResults(1)
					->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);
				$result += $result[0];
				unset($result[0]);

				return $result;
			});
		}

		return $this->cGroups[$groupId];
	}

	public function getGroupByUser(int $userId): array
	{
		if (!array_key_exists($userId, $this->cGroupsByUsers)) {
			$this->cGroupsByUsers[$userId] = $this->em->getRepository(Customer::class)->createQueryBuilder('c')
				->select('cg.id')
				->innerJoin('c.groupCustomers', 'cg')
				->where('c.user = :user')
				->setParameter('user', $userId)
				->setMaxResults(1)
				->getQuery()
				->enableResultCache(600)
				->getArrayResult()[0] ?? [];
		}

		return $this->cGroupsByUsers[$userId];
	}
}
