<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Helpers;

use Core\Model\Dao\SiteDomain;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\OrderStatus as EshopOrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Utils\Strings;
use Nette\Application\LinkGenerator;
use Spatie\SchemaOrg\BaseType;
use Spatie\SchemaOrg\Contracts\OrderStatusContract;
use Spatie\SchemaOrg\ItemAvailability;
use Spatie\SchemaOrg\Order as SchemaOrder;
use Spatie\SchemaOrg\Schema;

class SchemaGenerator
{
	protected LinkGenerator $linkGenerator;

	public function __construct(
		LinkGenerator $linkGenerator
	)
	{
		$this->linkGenerator = $linkGenerator;
	}

	public function getSchemaForEmail(BaseType $schema): string
	{
		$result = $schema->toScript();

		return $result;
	}

	public function generateOrderSchema(
		Order        $order,
		?SiteDomain  $siteDomain = null,
		?Seller      $seller = null,
		?OrderStatus $newestStatus = null,
		?string      $trackingNumber = null,
		?string      $trackingUrl = null
	): ?SchemaOrder
	{
		if (!EshopOrdersConfig::load('order.allowRichSnippets')) {
			return null;
		}

		try {
			$addrInv = $order->getAddressInvoice();

			$newestStatus = $newestStatus ?: $order->getNewestOrderStatus();
			$status       = $newestStatus ? $newestStatus->getStatus() : null;
			$createdTime  = $order->getCreatedTime();

			if (!$addrInv || !$status || !$createdTime) {
				return null;
			}

			$parcelDelivery = Schema::parcelDelivery()
				->name($order->getSpedition() ? $order->getSpedition()->getName() : '')
				->deliveryAddress(Schema::postalAddress()
					->streetAddress($addrInv->getStreet())
					->addressLocality($addrInv->getCity())
					->postalCode($addrInv->getPostal())
					->addressCountry(Strings::upper($addrInv->getCountry() ? $addrInv->getCountry()->getId() : 'CZ'))
				);

			if ($trackingNumber) {
				$parcelDelivery->trackingNumber($trackingNumber);
			}

			if ($trackingUrl) {
				$parcelDelivery->trackingUrl($trackingUrl);
			}

			$schemaOrder = Schema::order()
				->orderNumber((string) $order->getId())
				->orderStatus(self::getOrderStatusSchema($status->getId()))
				->orderDate($createdTime)
				->acceptedOffer(
					array_values(array_map(static function(OrderItem $item) use ($order, $siteDomain) {
						$text = $item->getOrderItemText();

						$img = $item->getProduct() && $item->getProduct()->getGallery() && $item->getProduct()->getGallery()->getCoverImage()
							? 'https://' . $siteDomain->getDomain() . $item->getProduct()->getGallery()->getCoverImage()->getFilePath()
							: null;

						return Schema::offer()
							->itemOffered(
								Schema::product()
									->name($text ? $text->getName() : '')
									->sku($item->getCode1())
									->image(Schema::imageObject()->url($img))
									->offers(
										Schema::offer()
											->price(number_format($item->getPrice(true), 2, '.', ''))
											->priceCurrency($order->getCurrencyCode())
											->availability(Schema::itemAvailability()::InStock)
									)
							)
							->price(number_format($item->getPrice(true), 2, '.', ''))
							->priceCurrency($order->getCurrencyCode())
							->eligibleQuantity(Schema::quantitativeValue()->value($item->getQuantity()));
					}, $order->getOrderItems()->toArray()))
				)
				->customer(
					Schema::person()
						->name($addrInv->getName())
						->email($addrInv->getEmail())
				)
				->seller(Schema::organization()
					->name($seller->name ?? '')
					->url($siteDomain ? 'https://' . $siteDomain->getDomain() : '')
				)->orderDelivery($parcelDelivery)
				->merchant(Schema::organization()
					->name($seller->name ?? '')
				)->setProperty('price', number_format($order->getPrice(true), 2, '.', ''))
				->setProperty('priceCurrency', $order->getCurrencyCode());

			foreach ($order->getOrderDiscounts() as $discount) {
				$schemaOrder->discountCode($discount->getCode());
			}

			if (!$order->isPaid) {
				$schemaOrder->paymentUrl($this->linkGenerator->link('EshopOrders:Front:Payment:pay', ['orderIdent' => $order->getIdent()]));
			}
		} catch (\Exception $e) {
			return null;
		}

		return $schemaOrder;
	}

	/**
	 * @return OrderStatusContract
	 */
	public static function getOrderStatusSchema(string $status)
	{
		switch ($status) {
			case EshopOrderStatus::STATUS_RESERVED:
			case EshopOrderStatus::STATUS_CREATED:
			case EshopOrderStatus::STATUS_PROCESSING:
			case EshopOrderStatus::STATUS_IS_PAID:
				return Schema::orderStatus()::OrderProcessing;
			case EshopOrderStatus::STATUS_CANCELED:
				return Schema::orderStatus()::OrderCancelled;
			case EshopOrderStatus::STATUS_FINISHED:
				return Schema::orderStatus()::OrderDelivered;
			case EshopOrderStatus::STATUS_SPEDITION:
				return Schema::orderStatus()::OrderInTransit;
			case EshopOrderStatus::STATUS_WAITING_FOR_PAYMENT:
				return Schema::orderStatus()::OrderPaymentDue;
			case EshopOrderStatus::STATUS_READY_FOR_PICKUP:
				return Schema::orderStatus()::OrderPickupAvailable;
		}

		return Schema::orderStatus()::OrderProcessing;
	}
}
