<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\FrontModule\Model\Dao\ProductReview;
use EshopOrders\Model\Entities\OrderItemReview;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;

class ProductReviews extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = OrderItemReview::class;

	public function __construct(
		protected EshopOrdersCache $eshopOrdersCache,
	)
	{
	}

	/**
	 * @return ProductReview[]
	 */
	public function getByProduct(int $productId): array
	{
		$key = EshopOrdersCache::cacheNamespace . '/' . $productId;

		return $this->eshopOrdersCache->getCache()->load($key, function(&$dep) use ($productId) {
			$dep                = [
				Cache::EXPIRATION => '1 year',
			];
			$dep[Cache::Tags][] = 'productReviews/' . $productId;

			$qb = $this->getEr()->createQueryBuilder('oir');
			$qb->select('oir.created, oir.rating, oir.description')
				->addSelect('COALESCE(oir.customer, TRIM(CONCAT(u.name, \' \', u.lastname))) as customer')
				->addSelect('CASE WHEN oir.status = \'top\' THEN 0 ELSE 1 END as hidden sort')
				->join('oir.orderItem', 'oi')
				->join('oi.order', 'o')
				->leftJoin('o.customer', 'c')
				->leftJoin('c.user', 'u')
				->andWhere($qb->expr()->in('oir.status', [
					OrderItemReview::STATUS_PUBLISHED,
					OrderItemReview::STATUS_TOP,
				]))
				->andWhere($qb->expr()->eq('oi.product', $productId))
				->addOrderBy('sort')
				->addOrderBy('oir.created', 'desc');

			$result = [];

			foreach ($qb->getQuery()->getArrayResult() as $data) {
				$result[] = new ProductReview($data['customer'], $data['created'], (float) $data['rating'], $data['description']);
			}

			return $result;
		});
	}

}
