<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\EventDispatcher;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Components\Cart\CartPreview;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\Entities\CartItem;
use EshopOrders\Model\EshopOrdersConfig;
use Gallery\FrontModule\Model\Dao\Image;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\Presenter;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CartSubscriber implements EventSubscriberInterface
{
	public static array $cartChanges = [
		'added'   => [],
		'updated' => [],
		'removed' => [],
		'isNew'   => [],
	];

	protected ?Presenter $presenter = null;

	public function __construct(
		protected Carts           $cartsService,
		protected ProductsFacade  $productsFacade,
		protected Tags            $tagsService,
		protected Translator      $translator,
		protected LinkGenerator   $linkGenerator,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::startup'      => 'frontPresenterStartup',
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
			'eshopOrders.cartAddItem'                => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'             => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'             => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems'           => ['onFillDaoItems', 100],
		];
	}

	public function frontPresenterStartup(ControlEvent $event): void
	{
		/** @var AbstractPresenter $presenter */
		$presenter       = $event->control->getPresenter();
		$this->presenter = $presenter;

		$callback = function() use ($presenter): void {
			/** @var CartPreview $cartPreview */
			$cartPreview = $presenter->getComponent('cartPreview', false);
			if ($cartPreview) {
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCount', (string) $cartPreview->getItemsCountFormatted());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCountRaw', (string) $cartPreview->getItemsCountRaw());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsPrice', (string) $cartPreview->getPriceFormatted());
			}

			$presenter->template->cart = $cartPreview->getCart();
			$presenter->redrawControl('cartMessage');
		};

		$this->eventDispatcher->addListener('eshopOrders.cartAddItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartRemoveItem', $callback);
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (self::$cartChanges !== []) {
			$message = null;
			$event->control->redrawControl('orderCartDetail');

			if (EshopOrdersConfig::load('orderCartDetail.messages.itemAdded.enable', true) && !empty(self::$cartChanges['added'])) {
				$message = 'itemAdded';
			}

			if (!empty(self::$cartChanges['updated'])) {
				$message = 'itemUpdated';
			}

			if (!empty(self::$cartChanges['removed'])) {
				$message = 'itemRemoved';
			}

			if ($message && method_exists($event->control, 'flashMessageSuccess')) {
				$event->control->flashMessageSuccess('eshopOrdersFront.cart.' . $message);
			}

			$event->control->redrawControl('flashes');
		}
	}

	public function onAddItem(AddedCartItemEvent $event): void
	{
		$item = &$event->item;

		if ($item->productId) {
			$product = $this->productsFacade->getProduct($item->productId);
			if (!$product || (!$item->ignoreValidation && !$product->canAddToCart)) {
				return;
			}
		}

		$result = $this->cartsService->addItem($item);

		if ($result === 1) {
			self::$cartChanges['added'][] = $item;
		}
	}

	public function onUpdateItem(UpdatedCartItemEvent $event): void
	{
		$result = $this->cartsService->updateItemQuantity((int) $event->itemId, $event->quantity, $event->moreData);

		if ($result === 1) {
			self::$cartChanges['updated'][] = $event->itemId;
		} else if ($result === 0) {
			self::$cartChanges['removed'][] = $event->itemId;
		}
	}

	public function onRemoveItem(RemovedCartItemEvent $event): void
	{
		self::$cartChanges['removed'][] = $event->itemId;
		$this->cartsService->removeItem($event->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		/** @var CartItem[] $cartItems */
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems) {
			return;
		}

		$productIds = array_map(static fn(CartItem $ci): ?int => $ci->getProductId(), $cartItems);

		foreach ($cartItems as $ci) {
			if ($ci->getMoreDataValue('productId')) {
				$productIds[] = (int) $ci->getMoreDataValue('productId');
			}
		}

		$products                     = $productIds !== [] ? $this->productsFacade->getProducts($productIds) : [];
		ProductsFacade::$loadFeatures = true;

		$items  = [];
		$childs = [];
		foreach ($cartItems as $ci) {
			$product = $products[$ci->getProductId()] ?? null;

			if ($product && !$product->canAddToCart && !$ci->getMoreDataValue('isCustom')) {
				continue;
			}

			if ($product) {
				$cartItem = new Dao\CartItem($ci->name ?: (string) $product->getName(), $product->getId(), $product->getPrice());
				$cartItem->setProduct($product)
					->setLink($product->link)
					->setVatRate($product->vatRate ?: 21);
				$cartItem->weight = $product->weight ?: 0;
			} else {
				$cartItem = new Dao\CartItem((string) $ci->name, $ci->productId, (float) $ci->price);
				$cartItem->setVatRate($ci->vatRate ?? 21);
				$cartItem->weight = 0;

				if ($ci->getMoreDataValue('productId') && isset($products[$ci->getMoreDataValue('productId')])) {
					$cartItem->setProduct($products[$ci->getMoreDataValue('productId')]);
				}
			}

			$cartItem
				->setId($ci->getId())
				->setQuantity($ci->quantity);

			$cartItem->ident = (string) $ci->getIdent();
			$cartItem->setData($ci->getMoreData());

			/** @var float $extraPrice */
			$extraPrice                    = $ci->getMoreDataValue('extraPrice') ?: 0;
			$cartItem->priceInBaseCurrency = ($product->priceInBaseCurrency ?? (float) $ci->price) + $extraPrice;

			if ($product) {
				$cartItem->discountDisabled              = $product->discountDisabled;
				$cartItem->disableCalculateFreeSpedition = $product->disableCalculateFreeSpedition;
				$cartItem->verifyAge                     = $product->verifyAge;

				if ($product->getGallery() && $product->getGallery()->getCover()) {
					$cartItem->setImage($product->getGallery()->getCover());
				}

				if (EshopOrdersConfig::load('allowFreeFrom')) {
					$tagFree = $product->getTag('freeDelivery');

					if ($tagFree && !$tagFree->isAuto) {
						$cartItem->freeDelivery = true;
					}
				}

				if ($ci->image) {
					$cartItem->image = new Image();
					$cartItem->image->setFile($ci->image);
				}
			}

			if ($ci->getParent()) {
				$childs[$ci->getParent()->getId()][$ci->getId()] = $cartItem;
			} else {
				$items[$ci->getId()] = $cartItem;
			}
		}

		foreach ($childs as $parentId => $parentChilds) {
			foreach ($parentChilds as $child) {
				/** @var Dao\CartItem $child */
				if (isset($items[$parentId])) {
					$itemProduct = $items[$parentId]->product;

					if ($itemProduct && $itemProduct->package) {
						$child->priceInBaseCurrency = 0;
						$child->price               = 0;
					}

					$items[$parentId]->addChild($child);
				}
			}
		}

		$this->cartsService->cDaoItems = $items;
	}
}
