<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table('eshop_orders__cart_item')]
#[ORM\Entity]
class CartItem
{
	use TId;

	/**
	 * Identifikator slozeny z id produktu, varianty, a pripadne dalsich parametru, ktere maji vliv na cenu
	 */
	#[ORM\Column(name: 'ident', type: Types::STRING, nullable: false)]
	protected string $ident;

	#[ORM\Column(name: 'name', type: Types::STRING, length: 255, nullable: true)]
	public ?string $name = null;

	/**
	 * @var double|string
	 */
	#[ORM\Column(name: 'price', type: Types::DECIMAL, precision: 10, scale: 3, nullable: true)]
	public $price;

	#[ORM\Column(name: 'image', type: Types::STRING, length: 255, nullable: true)]
	public ?string $image = null;

	#[ORM\Column(name: 'product_id', type: Types::INTEGER, nullable: true)]
	public ?int $productId = null;

	#[ORM\Column(name: 'quantity', type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $quantity;

	#[ORM\Column(name: 'vat_rate', type: Types::SMALLINT, nullable: true)]
	public ?int $vatRate = null;

	#[ORM\JoinColumn(name: 'cart_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Cart::class, inversedBy: 'cartItems')]
	public Cart $cart;

	#[ORM\Column(name: 'more_data', type: 'array', nullable: true)]
	protected ?array $moreData = [];

	/** @var ArrayCollection<self> */
	#[ORM\OneToMany(mappedBy: 'parent', targetEntity: CartItem::class)]
	protected Collection $children;

	#[ORM\JoinColumn(name: 'parent_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: CartItem::class, inversedBy: 'children')]
	protected ?CartItem $parent = null;

	public function __construct(string $ident, ?int $productId, Cart $cart, int $quantity = 1)
	{
		$this->ident     = $ident;
		$this->productId = $productId;
		$this->quantity  = $quantity;
		$this->cart      = $cart;
		$this->moreData  = [];
		$this->children  = new ArrayCollection();

		//Při vytvoření nové položky košíku je potřeba ji přidat rovnou do Entity košíku. Jelikož Proxy Entity ji po vytvoření neobsahuje
		if ($ident) {
			$this->cart->cartItems->set($ident, $this);
		} else {
			$this->cart->addCartItem($this);
		}
	}

	public function getIdent(): string { return $this->ident; }

	public function getProductId(): ?int { return $this->productId; }

	public function getMoreData(): array { return $this->moreData ?? []; }

	/** @return mixed */
	public function getMoreDataValue(string $key)
	{
		return $this->moreData[$key] ?? null;
	}

	public function setMoreData(array $data): self
	{
		if ($this->moreData['note'] ?? null) {
			$data['note'] = $this->moreData['note'];
		}

		$this->moreData = $data;

		return $this;
	}

	/** @param mixed $value */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}

	/** @return Collection<self> */
	public function getChildren() { return $this->children; }

	public function getParent(): ?CartItem { return $this->parent; }

	public function setParent(CartItem $parent): void
	{
		$this->parent = $parent;
		$parent->getChildren()->set($this->id, $this);
	}
}
