<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table('eshop_orders__group_customers')]
#[ORM\Entity]
class GroupCustomers
{
	public const typeB2B = 'B2B';
	public const typeB2C = 'B2C';

	use TId;

	#[ORM\Column(name: 'short', type: Types::STRING, length: 60)]
	public string $short = '';

	#[ORM\Column(name: 'name', type: Types::STRING, length: 255)]
	public string $name = '';

	#[ORM\Column(name: 'is_published', type: Types::SMALLINT, length: 1)]
	public int $isPublished = 0;

	/** @var Collection<Customer> */
	#[ORM\OneToMany(targetEntity: Customer::class, mappedBy: 'groupCustomers', cascade: ['all'], indexBy: 'id')]
	protected Collection $customers;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'products_sale', type: Types::DECIMAL, precision: 10, scale: 2, nullable: false, options: ['default' => 0])]
	protected $productsSale;

	/** @var Collection<GroupParam> */
	#[ORM\OneToMany(targetEntity: GroupParam::class, mappedBy: 'group', indexBy: 'key')]
	protected Collection $params;

	#[ORM\Column(name: 'disable_measuring_codes', type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $disableMeasuringCodes = 0;

	#[ORM\Column(name: 'products_sale_use_group_price', type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $productsSaleUseGroupPrice = 0;

	#[ORM\Column(name: 'access_only_to_allowed_categories', type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $accessOnlyToAllowedCategories = 0;

	#[ORM\JoinColumn(name: 'inherit_prices_group', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: GroupCustomers::class)]
	public ?GroupCustomers $inheritPricesGroup = null;

	#[ORM\JoinColumn(name: 'use_products_sale_on_group', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: GroupCustomers::class)]
	public ?GroupCustomers $useProductsSaleOnGroup = null;

	#[ORM\Column(name: 'multiple_to_order', type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	public ?int $multipleToOrder = null;

	#[ORM\Column(name: 'invoice_copy_email', type: Types::STRING, length: 255, nullable: true)]
	public ?string $invoiceCopyEmail = null;

	#[ORM\Column(name: '`type`', type: Types::STRING, length: 25, options: ['default' => 'B2C'])]
	public string $type = 'B2C';

	public function __construct(string $short, string $name)
	{
		$this->short        = $short;
		$this->name         = $name;
		$this->productsSale = 0;
		$this->customers    = new ArrayCollection();
		$this->params       = new ArrayCollection();
	}

	public function getProductsSale(): float { return (float) $this->productsSale; }

	public function setProductsSale(float $value): self
	{
		$this->productsSale = (string) Strings::formatEntityDecimal($value);

		return $this;
	}

	/** @return Collection<GroupParam> */
	public function getParams() { return $this->params; }

	public static function getTypes(): array
	{
		return [
			self::typeB2C => 'eshop.orders.groupCustomers.type.b2c',
			self::typeB2B => 'eshop.orders.groupCustomers.type.b2b',
		];
	}
}
