<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use DateTime;
use DateTimeInterface;
use Core\Model\Entities\TId;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice\InvoiceData;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Listeners\InvoiceListener;

#[ORM\Table('eshop_orders__invoice')]
#[ORM\Entity]
#[ORM\EntityListeners([InvoiceListener::class])]
class Invoice
{
	use TId;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTime $dueDate = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTime $duzp = null;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $ident;

	#[ORM\OneToOne(mappedBy: 'invoice', targetEntity: Order::class, cascade: ['persist'])]
	public ?Order $order = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: false)]
	public ?DateTime $createdAt;

	#[ORM\OneToOne(mappedBy: 'invoice', targetEntity: InvoiceData::class, cascade: ['persist'])]
	public ?InvoiceData $invoiceData = null;

	#[ORM\JoinColumn(name: 'seller_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Seller::class)]
	public Seller $seller;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	protected ?string $externalInvoiceFile = null;

	#[ORM\Column(type: Types::TEXT, nullable: true, options: ['default' => null])]
	public ?string $note = null;

	public function __construct(?DateTime $dueDate, string $ident, Seller $seller)
	{
		$this->dueDate   = $dueDate;
		$this->ident     = $ident;
		$this->seller    = $seller;
		$this->createdAt = new DateTime();
	}

	public function setExternalInvoiceFile(?string $externalInvoiceFile): void
	{
		if ($externalInvoiceFile === null) {
			$this->externalInvoiceFile = null;

			return;
		}

		if (file_exists(WWW_DIR . $externalInvoiceFile)) {
			$this->externalInvoiceFile = $externalInvoiceFile;
		}
	}

	public function getExternalInvoiceFile(): ?string
	{
		if ($this->externalInvoiceFile !== null && file_exists(WWW_DIR . $this->externalInvoiceFile)) {
			return $this->externalInvoiceFile;
		}

		return null;
	}

	public function isExternal(): bool
	{
		return $this->externalInvoiceFile !== null;
	}

	public function getDUZP(): ?DateTimeInterface
	{
		if ($this->duzp) {
			return $this->duzp;
		}

		if (EshopOrdersConfig::load('invoice.document.separatedDUZPCreated')) {
			return $this->order->getCreatedTime() ?: $this->createdAt;
		}

		return $this->createdAt;
	}
}
