<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use DateTime;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderItemReviewsHelper;

#[ORM\Table(name: 'eshop_orders__order_item_review')]
#[ORM\Entity]
class OrderItemReview
{
	public const STATUS_NEW           = 'new';
	public const STATUS_PUBLISHED     = 'published';
	public const STATUS_NOT_PUBLISHED = 'notPublished';
	public const STATUS_TOP           = 'top';
	public const STATUSES             = [
		self::STATUS_NEW,
		self::STATUS_PUBLISHED,
		self::STATUS_NOT_PUBLISHED,
		self::STATUS_TOP,
	];

	use TId;

	#[ORM\JoinColumn(referencedColumnName: 'id', nullable: false, onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: OrderItem::class, inversedBy: 'orderItemReview')]
	public OrderItem $orderItem;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, options: ['default' => 'CURRENT_TIMESTAMP'])]
	public DateTime $created;

	/**
	 * @var float|string
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 1, nullable: false)]
	public $rating;

	#[ORM\Column(name: 'description', type: Types::TEXT, nullable: true)]
	public ?string $description;

	#[ORM\Column(type: Types::STRING, nullable: false, options: ['default' => 'new'])]
	public string $status = self::STATUS_NEW;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $customer = null;

	public function __construct(OrderItem $orderItem, DateTime $created, float $rating, ?string $description)
	{
		$this->orderItem   = $orderItem;
		$this->created     = $created;
		$this->rating      = (string) Strings::formatEntityDecimal($rating);
		$this->description = $description;
	}

	public function getRating(): float
	{
		return (float) Strings::formatEntityDecimal($this->rating);
	}

	/**
	 * @return float[]
	 */
	public function getRatingAsArray(bool $withBlanks = false): array
	{
		return OrderItemReviewsHelper::ratingToArray($this->getRating(), $withBlanks);
	}

	public function getMaxRating(): float
	{
		return (float) EshopOrdersConfig::loadInt('orderItemReviews.maxRating');
	}

}
