<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;

/** podobne jako order_item, ale samostatna tabulka
 */
#[ORM\Table('eshop_orders__order_payment')]
#[ORM\Entity]
class OrderPayment
{
	use TId;

	#[ORM\JoinColumn(name: 'payment_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Payment::class)]
	public ?Payment $payment = null;

	#[ORM\Column(name: 'name', type: Types::STRING)]
	protected ?string $name = null;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'price', type: Types::DECIMAL, precision: 10, scale: 2)]
	protected $price;

	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\OneToOne(inversedBy: 'payment', targetEntity: Order::class)]
	public ?Order $order = null;

	#[ORM\Column(name: 'vat_rate', type: Types::SMALLINT, nullable: true)]
	protected ?int $vatRate = null;

	public ?PriceCalculatorItem $priceCalculatorItem         = null;
	public ?PriceCalculatorItem $priceCalculatorItemCurrency = null;

	public function __construct(?Payment $payment, Order $order)
	{
		$this->payment = $payment;
		$this->name    = $payment ? $payment->getName() : '';
		$this->order   = $order;

		if ($payment) {
			$this->price = $payment->getPrice();
		}
	}

	public function getIdent(): ?string { return $this->getPayment() ? $this->getPayment()->getIdent() : null; }

	public function getPaymentId(): ?int { return $this->payment ? $this->payment->getId() : null; }

	public function getPayment(): ?Payment { return $this->payment; }

	public function setPayment(Payment $payment): self
	{
		$this->payment = $payment;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPriceRaw(): float { return (float) $this->price; }

	public function getPrice(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithoutVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): ?Order { return $this->order; }

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?: 21); }
}
