<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopOrders\Model\Entities\CarrierHistory;
use EshopOrders\Model\Entities\Order;
use Exception;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class CarrierListener implements EventSubscriberInterface
{
	protected static bool $saved = false;

	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'orderSubscriber.orderSpeditionFormSuccess' => ['speditionChanged'],
		];
	}

	/**
	 * @param object $entity hlavni entita z dopravcu (UlozenkaOrder, atd.)
	 */
	#[ORM\PostPersist]
	#[ORM\PostUpdate]
	public function postHandler($entity, LifecycleEventArgs $event): void
	{
		$em = $this->em;

		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($entity);

		if ($changeSet && isset($changeSet['numberPackage'])) {
			if (!method_exists($entity, 'getOrder')) {
				return;
			}

			/** @var Order $order */
			$order = $entity->getOrder();

			$packages = [];
			if (method_exists($entity, 'getAllNumberPackages')) {
				$packages = $entity->getAllNumberPackages();
			} else if (property_exists($entity, 'numberPackage')) {
				$packages[] = $entity->numberPackage;
			}

			$this->speditionChanged(new Event(['order' => $order, 'numberPackages' => $packages]));
			static::$saved = true;
		}
	}

	public function speditionChanged(Event $event): void
	{
		if (static::$saved) {
			return;
		}

		$this->save($event->data['numberPackages'], $event->data['order']);
		static::$saved = true;
	}

	protected function save(array $packages, Order $order): void
	{
		$ch = new CarrierHistory($order, $order->getSpedition()->getName(), $packages !== [] ? implode(', ', $packages) : null);

		try {
			$this->em->persist($ch);
			$this->em->flush();
		} catch (Exception $e) {
			Debugger::log($e, 'carrier-listener');
		}
	}

}
