<?php declare(strict_types = 1);

namespace EshopOrders\Model\PriceCalculator;

use Brick\Money\Money;
use EshopOrders\Model\Helpers\OrderHelper;

class PriceCalculator
{
	/** @var PriceCalculatorItem[] */
	protected array $items = [];

	/** @var PriceCalculatorDiscount[] */
	public array     $discountCodes = [];
	protected ?Money $totalDiscount = null;

	protected ?PriceCalculatorItem $spedition = null;
	protected ?PriceCalculatorItem $payment   = null;

	protected ?Money $cWithoutVatTotal                = null;
	protected ?Money $cWithoutVatTotalWithoutDiscount = null;
	protected ?Money $cWithoutVatItems                = null;
	protected ?Money $cWithoutVatItemsWithoutDiscount = null;
	protected ?Money $cWithVatTotal                   = null;
	protected ?Money $cWithVatTotalWithoutDiscount    = null;
	protected ?Money $cWithVatItems                   = null;
	protected ?Money $cWithVatItemsWithoutDiscount    = null;
	protected ?array $cDiscountsByVat                 = null;
	protected ?Money $cTotalVat                       = null;
	protected ?array $cVatBreakdown                   = null;
	protected ?int   $cItemsCount                     = null;

	public function __construct(protected string $currency)
	{
		$this->totalDiscount = Money::of(0, $this->currency);
	}

	public function clearCache(): void
	{
		$this->cWithoutVatTotal = null;
		$this->cDiscountsByVat  = null;
		$this->cWithVatTotal    = null;
		$this->cTotalVat        = null;
		$this->cVatBreakdown    = null;
		$this->cItemsCount      = null;
	}

	public function addItem(string $key, PriceCalculatorItem $item): void
	{
		$this->items[$key] = $item;

		$this->clearCache();
	}

	public function addDiscountCode(string $key, PriceCalculatorDiscount $discount): void
	{
		$this->discountCodes[$key] = $discount;

		$this->clearCache();
	}

	public function getSpedition(): ?PriceCalculatorItem
	{
		return $this->spedition;
	}

	public function setSpedition(PriceCalculatorItem $spedition): void
	{
		$this->spedition = $spedition;

		$this->clearCache();
	}

	public function getPayment(): ?PriceCalculatorItem
	{
		return $this->payment;
	}

	public function setPayment(PriceCalculatorItem $payment): void
	{
		$this->payment = $payment;

		$this->clearCache();
	}

	public function getCurrency(): string
	{
		return $this->currency;
	}

	public function changeItemQuantity(string $key, int $quantity): void
	{
		if (!isset($this->items[$key])) {
			return;
		}

		$this->items[$key]->setQuantity($quantity);

		$this->clearCache();
	}

	public function removeItem(string $key): void
	{
		if (!isset($this->items[$key])) {
			return;
		}

		unset($this->items[$key]);

		$this->clearCache();
	}

	/**
	 * @return PriceCalculatorItem[]
	 */
	public function getItems(): array
	{
		return $this->items;
	}

	/**
	 * Vrátí počet produktů v košíku
	 */
	public function getItemsCount(): int
	{
		if ($this->cItemsCount === null) {
			$this->cItemsCount = 0;

			foreach ($this->items as $item) {
				$this->cItemsCount += $item->getQuantity();
			}
		}

		return $this->cItemsCount;
	}

	/**
	 * Vrátí celkovou cenu bez DPH polozek
	 */
	public function getWithoutVatItems(): Money
	{
		if (!$this->cWithoutVatItems instanceof Money) {
			$this->cWithoutVatItems = Money::of(0, $this->currency);

			// Polozky
			foreach ($this->items as $item) {
				$this->cWithoutVatItems = $this->cWithoutVatItems->plus($item->getTotalWithoutVat());
			}

			// Slevy
			foreach ($this->getDiscountsByVat() as $prices) {
				$this->cWithoutVatItems = $this->cWithoutVatItems->plus($prices['withoutVat']);
			}
		}

		return $this->cWithoutVatItems;
	}

	/**
	 * Vrátí celkovou cenu bez DPH polozek bez slev
	 */
	public function getWithoutVatItemsWithoutDiscount(): Money
	{
		if (!$this->cWithoutVatItemsWithoutDiscount instanceof Money) {
			$this->cWithoutVatItemsWithoutDiscount = Money::of(0, $this->currency);

			// Polozky
			foreach ($this->items as $item) {
				$this->cWithoutVatItemsWithoutDiscount = $this->cWithoutVatItemsWithoutDiscount->plus($item->getTotalWithoutVat());
			}
		}

		return $this->cWithoutVatItemsWithoutDiscount;
	}

	/**
	 * Vrátí celkovou cenu bez DPH (polozky, doprava, platba)
	 */
	public function getWithoutVatTotal(): Money
	{
		if (!$this->cWithoutVatTotal instanceof Money) {
			$this->cWithoutVatTotal = $this->getWithoutVatItems();

			// Doprava
			if ($this->spedition instanceof PriceCalculatorItem) {
				$this->cWithoutVatTotal = $this->cWithoutVatTotal->plus($this->spedition->getTotalWithoutVat());
			}

			// Platba
			if ($this->payment instanceof PriceCalculatorItem) {
				$this->cWithoutVatTotal = $this->cWithoutVatTotal->plus($this->payment->getTotalWithoutVat());
			}
		}

		return $this->cWithoutVatTotal;
	}

	/**
	 * Vrátí celkovou cenu bez DPH bez slev (polozky, doprava, platba)
	 */
	public function getWithoutVatTotalWithoutDiscounts(): Money
	{
		if ($this->cWithoutVatTotalWithoutDiscount === null) {
			$this->cWithoutVatTotalWithoutDiscount = $this->getWithoutVatItemsWithoutDiscount();

			// Doprava
			if ($this->spedition) {
				$this->cWithoutVatTotalWithoutDiscount = $this->cWithoutVatTotalWithoutDiscount->plus($this->spedition->getTotalWithoutVat());
			}

			// Platba
			if ($this->payment) {
				$this->cWithoutVatTotalWithoutDiscount = $this->cWithoutVatTotalWithoutDiscount->plus($this->payment->getTotalWithoutVat());
			}
		}

		return $this->cWithoutVatTotalWithoutDiscount;
	}

	/**
	 * Vrátí slevy podle DPH.
	 *
	 * @return array<int, array<string, Money>>
	 */
	public function getDiscountsByVat(): array
	{
		if ($this->cDiscountsByVat === null) {
			$this->cDiscountsByVat = [];

			if (empty($this->discountCodes)) {
				return $this->cDiscountsByVat;
			}

			// Celkova cena polozek pro slevy
			$itemsPrice = Money::of(0, $this->currency);
			// Celkova cena vsech polozek
			$subTotal = Money::of(0, $this->currency);
			// Rozdeleni slev podle DPH
			$dphRates = [];
			// Celkova sleva
			$totalSale = Money::of(0, $this->currency);

			foreach ($this->items as $item) {
				// Celkova cena polozek
				$subTotal = $subTotal->plus($item->getTotalWithoutVat());

				if ($item->isDiscountApplicable()) {
					// Cena polozek pro slevy
					$itemsPrice = $itemsPrice->plus($item->getTotalWithVat());

					$vat = $item->getVatRate();
					if (!isset($dphRates[$vat])) {
						$dphRates[$vat] = Money::of(0, $this->currency);
					}

					// Rozdeleni podle DPH
					$dphRates[$vat] = $dphRates[$vat]->plus($item->getTotalWithVat());
				}
			}

			// Celkova sleva
			foreach ($this->discountCodes as $discount) {
				// Hodnota slevy je v minusu
				$totalSale = $totalSale->plus($discount->getValue()->abs());
			}

			// Rozdeleni slev podle DPH
			foreach (OrderHelper::calculateVatWithSalesMoney($itemsPrice, $totalSale, $dphRates) as $vat => $v) {
				if (!isset($this->cDiscountsByVat[$vat])) {
					$this->cDiscountsByVat[$vat]['withoutVat'] = Money::of(0, $this->currency);
					$this->cDiscountsByVat[$vat]['total']      = Money::of(0, $this->currency);
				}
				$this->cDiscountsByVat[$vat]['withoutVat'] = $this->cDiscountsByVat[$vat]['withoutVat']->plus($v['withoutVat']);
				$this->cDiscountsByVat[$vat]['total']      = $this->cDiscountsByVat[$vat]['total']->plus($v['total']);
			}
		}

		return $this->cDiscountsByVat;
	}

	/**
	 * Vrátí celkovou cenu s DPH polozek
	 */
	public function getWithVatItems(): Money
	{
		if (!$this->cWithVatItems instanceof Money) {
			$this->cWithVatItems = Money::of(0, $this->currency);

			// Polozky
			foreach ($this->items as $item) {
				$this->cWithVatItems = $this->cWithVatItems->plus($item->getTotalWithVat());
			}

			// Slevy
			foreach ($this->discountCodes as $discount) {
				$this->cWithVatItems = $this->cWithVatItems->minus($discount->getValue()->abs());
			}
		}

		return $this->cWithVatItems;
	}

	/**
	 * Vrátí celkovou cenu s DPH polozek bez slev
	 */
	public function getWithVatItemsWithoutDiscount(): Money
	{
		if (!$this->cWithVatItemsWithoutDiscount instanceof Money) {
			$this->cWithVatItemsWithoutDiscount = Money::of(0, $this->currency);

			// Polozky
			foreach ($this->items as $item) {
				$this->cWithVatItemsWithoutDiscount = $this->cWithVatItemsWithoutDiscount->plus($item->getTotalWithVat());
			}
		}

		return $this->cWithVatItemsWithoutDiscount;
	}

	/**
	 * Vrátí celkovou cenu s DPH (polozky, doprava, platba)
	 */
	public function getWithVatTotal(): Money
	{
		if (!$this->cWithVatTotal instanceof Money) {
			$this->cWithVatTotal = $this->getWithVatItems();

			// Doprava
			if ($this->spedition instanceof PriceCalculatorItem) {
				$this->cWithVatTotal = $this->cWithVatTotal->plus($this->spedition->getTotalWithVat());
			}

			// Platba
			if ($this->payment instanceof PriceCalculatorItem) {
				$this->cWithVatTotal = $this->cWithVatTotal->plus($this->payment->getTotalWithVat());
			}
		}

		return $this->cWithVatTotal;
	}

	/**
	 * Vrátí celkovou cenu s DPH bez slev (polozky, doprava, platba)
	 */
	public function getWithVatTotalWithoutDiscounts(): Money
	{
		if ($this->cWithVatTotalWithoutDiscount === null) {
			$this->cWithVatTotalWithoutDiscount = $this->getWithVatItemsWithoutDiscount();

			// Doprava
			if ($this->spedition) {
				$this->cWithVatTotalWithoutDiscount = $this->cWithVatTotalWithoutDiscount->plus($this->spedition->getTotalWithVat());
			}

			// Platba
			if ($this->payment) {
				$this->cWithVatTotalWithoutDiscount = $this->cWithVatTotalWithoutDiscount->plus($this->payment->getTotalWithVat());
			}
		}

		return $this->cWithVatTotalWithoutDiscount;
	}

	/**
	 * Vrátí celkovou hodnotu DPH
	 */
	public function getTotalVat(): Money
	{
		if (!$this->cTotalVat instanceof Money) {
			$this->cTotalVat = Money::of(0, $this->currency);

			// DPH z rozlozeni
			foreach ($this->getVatBreakdown() as $prices) {
				$this->cTotalVat = $this->cTotalVat->plus($prices['rate']);
			}
		}

		return $this->cTotalVat;
	}

	/**
	 * Vrátí rozložení DPH.
	 *
	 * @return array<int, array<string, Money>>
	 */
	public function getVatBreakdown(): array
	{
		if ($this->cVatBreakdown === null) {
			$this->cVatBreakdown = [];

			// Inicializace pole
			$initVatBreakdown = function(int $vatRate): void {
				$this->cVatBreakdown[$vatRate]['withoutVat'] = Money::of(0, $this->currency);
				$this->cVatBreakdown[$vatRate]['total']      = Money::of(0, $this->currency);
			};

			// Polozky
			foreach ($this->items as $item) {
				$vatRate = $item->getVatRate();
				if (!isset($this->cVatBreakdown[$vatRate])) {
					$initVatBreakdown((int) $vatRate);
				}

				$this->cVatBreakdown[$vatRate]['withoutVat'] = $this->cVatBreakdown[$vatRate]['withoutVat']->plus($item->getTotalWithoutVat());
				$this->cVatBreakdown[$vatRate]['total']      = $this->cVatBreakdown[$vatRate]['total']->plus($item->getTotalWithVat());
			}

			// Doprava
			if ($this->spedition instanceof PriceCalculatorItem) {
				$vatRate = $this->spedition->getVatRate();
				if (!isset($this->cVatBreakdown[$vatRate])) {
					$initVatBreakdown((int) $vatRate);
				}

				$this->cVatBreakdown[$vatRate]['withoutVat'] = $this->cVatBreakdown[$vatRate]['withoutVat']->plus($this->spedition->getTotalWithoutVat());
				$this->cVatBreakdown[$vatRate]['total']      = $this->cVatBreakdown[$vatRate]['total']->plus($this->spedition->getTotalWithVat());
			}

			// Platba
			if ($this->payment instanceof PriceCalculatorItem) {
				$vatRate = $this->payment->getVatRate();
				if (!isset($this->cVatBreakdown[$vatRate])) {
					$initVatBreakdown((int) $vatRate);
				}

				$this->cVatBreakdown[$vatRate]['withoutVat'] = $this->cVatBreakdown[$vatRate]['withoutVat']->plus($this->payment->getTotalWithoutVat());
				$this->cVatBreakdown[$vatRate]['total']      = $this->cVatBreakdown[$vatRate]['total']->plus($this->payment->getTotalWithVat());
			}

			// Slevy
			foreach ($this->getDiscountsByVat() as $vat => $prices) {
				if (!isset($this->cVatBreakdown[$vat])) {
					$initVatBreakdown((int) $vat);
				}

				$this->cVatBreakdown[$vat]['withoutVat'] = $this->cVatBreakdown[$vat]['withoutVat']->plus($prices['withoutVat']);
				$this->cVatBreakdown[$vat]['total']      = $this->cVatBreakdown[$vat]['total']->plus($prices['total']);
			}

			// Vypocet hodnoty DPH
			foreach ($this->cVatBreakdown as $vat => $prices) {
				$this->cVatBreakdown[$vat]['rate'] = $prices['total']->minus($prices['withoutVat']);
			}
		}

		return $this->cVatBreakdown;
	}
}
