<?php declare(strict_types = 1);

namespace EshopOrders\Model\Statistics;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\Entities\EntityManagerDecorator;
use DateTime;
use DateTimeInterface;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Config;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopStock\DI\EshopStockExtension;
use EshopStock\Model\Entities\Supply;
use EshopStock\Model\Entities\SupplyProduct;

class BaseStatistics
{
	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	public function getBaseOrdersQB(
		DateTimeInterface $from,
		DateTimeInterface $to,
		?string           $siteIdent = null,
		array             $sites = []
	): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();

		$osCratedQb = $this->em->createQueryBuilder()->select('IDENTITY(os.order)')
			->from(OrderStatus::class, 'os')
			->where('os.status = :statusCrea')
			->andWhere('os.created >= :from')
			->andWhere('os.created <= :to');

		$query = $this->em->createQueryBuilder()
			->select('o.id, p.price as paymentPrice, s.price as speditionPrice, IDENTITY(o.site) as site, IDENTITY(o.customer) as customer, curr.code as currencyCode, curr.rate as currencyRate, curr.decimals as currencyDecimals, IDENTITY(o.site) as siteIdent, o.params')
			->from(Order::class, 'o')
			->join('o.payment', 'p')
			->join('o.spedition', 's')
			->leftJoin('o.currency', 'curr')
			->where($qb->expr()->in('o.id', $osCratedQb->getDQL()))
			->setParameters(new ArrayCollection([new Parameter('statusCrea', OrderStatus::STATUS_CREATED), new Parameter('from', $from), new Parameter('to', $to)]));

		if ($siteIdent) {
			$query->andWhere('o.site = :site')
				->setParameter('site', $siteIdent);
		} else if ($sites) {
			$query->andWhere('o.site IN (:sites)')
				->setParameter('sites', $sites);
		}

		return $query;
	}

	public function getSales(array $ids, bool $setAbsolute = false): array
	{
		$sales = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(od.order) as order, od.price')
				         ->from(OrderDiscount::class, 'od')
				         ->where('od.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$price                  = $setAbsolute ? abs((float) $row['price']) : (float) $row['price'];
				$sales[$row['order']][] = $price;
			}
		}

		return $sales;
	}

	public function getOrderItems(array $ids, ?callable $qbCallback = null, ?callable $dataCallback = null): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			$stockData = [];
			if (class_exists(EshopStockExtension::class) && class_exists(SupplyProduct::class) && Config::load('enablePurchasePrice')) {
				foreach ($this->em->createQueryBuilder()->select('IDENTITY(sp.product) as product, IDENTITY(sp.order) as order, sp.purchasePrice')
					         ->from(SupplyProduct::class, 'sp')
					         ->where('sp.order IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getScalarResult() as $row) {
					$stockData[$row['order']][$row['product']] = (float) $row['purchasePrice'];
				}
			}

			$qb = $this->em->createQueryBuilder()->select('IDENTITY(oi.order) as order, oi.quantity, oi.price, oi.vatRate, IDENTITY(oi.product) as product,
			    s.value as saleValue, s.type as saleType, oi.moreData')
				->from(OrderItem::class, 'oi')
				->where('oi.order IN (' . implode(',', $chunk) . ')')
				->leftJoin('oi.sales', 's');

			if (Config::load('enablePurchasePrice')) {
				$qb->addSelect('p.purchasePrice')
					->leftJoin('oi.product', 'p');
			}

			if ($qbCallback) {
				$qbCallback($qb);
			}

			foreach ($qb->getQuery()->getScalarResult() as $row) {
				$purchasePrice = $stockData[$row['order']][$row['product']] ?? null;
				if ($purchasePrice === null && isset($row['purchasePrice'])) {
					$purchasePrice = (float) $row['purchasePrice'];
				}

				$item = [
					'product'       => $row['product'],
					'price'         => (float) $row['price'],
					'purchasePrice' => $purchasePrice ?: 0,
					'quantity'      => (int) $row['quantity'],
					'vatRate'       => (int) $row['vatRate'],
					'saleValue'     => $row['saleValue'],
					'saleType'      => $row['saleType'],
					'moreData'      => $row['moreData'] ? unserialize($row['moreData']) : [],
				];

				if ($dataCallback) {
					$dataCallback($item, $row, $items);
				}

				$items[$row['order']][] = $item;
			}
		}

		return $items;
	}

	public function getPayments(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(op.order) as order, op.price, op.vatRate')
				         ->from(OrderPayment::class, 'op')
				         ->where('op.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$items[$row['order']][] = [
					'price'   => (float) $row['price'],
					'vatRate' => (int) $row['vatRate'],
				];
			}
		}

		return $items;
	}

	public function getSpeditions(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(os.order) as order, os.price, os.vatRate')
				         ->from(OrderSpedition::class, 'os')
				         ->where('os.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$items[$row['order']][] = [
					'price'   => (float) $row['price'],
					'vatRate' => (int) $row['vatRate'],
				];
			}
		}

		return $items;
	}

	public function getAddrInvoice(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			$ids = [];

			foreach ($this->em->getConnection()->executeQuery("SELECT id, address_delivery_id, address_invoice_id 
					FROM eshop_orders__order 
					WHERE id IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
				if ($row['address_invoice_id']) {
					$ids[$row['address_invoice_id']] = $row['id'];
				} else if ($row['address_delivery_id']) {
					$ids[$row['address_delivery_id']] = $row['id'];
				}
			}

			if (!empty($ids)) {
				foreach ($this->em->getConnection()->executeQuery("SELECT id, country_id, validated_vat_number, id_number, vat_number 
					FROM eshop_orders__order_address 
					WHERE id IN (" . implode(',', array_keys($ids)) . ")")->iterateAssociative() as $row) {
					$orderId = $ids[$row['id']];

					$items[$orderId] = [
						'country'            => $row['country_id'],
						'validatedVatNumber' => $row['validated_vat_number'],
						'idNumber'           => $row['id_number'],
						'vatNumber'          => $row['vat_number'],
					];
				}
			}
		}

		return $items;
	}

	public function getCancelled(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order')
				         ->from(OrderStatus::class, 'os')
				         ->where('os.order IN (' . implode(',', $chunk) . ')')
				         ->andWhere('os.status = :status')
				         ->setParameters(new ArrayCollection([new Parameter('status', OrderStatus::STATUS_CANCELED)]))->getQuery()->getScalarResult() as $row) {
				$items[$row['order']] = $row['order'];
			}
		}

		return $items;
	}

	/**
	 *
	 * @return array<int, DateTime>
	 */
	public function getCreated(DateTimeInterface $from, DateTimeInterface $to): array
	{
		$dates = [];
		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(os.order) as order, os.created')
			         ->from(OrderStatus::class, 'os')
			         ->andWhere('os.status = :statusCrea')
			         ->andWhere('os.created >= :from')
			         ->andWhere('os.created <= :to')
			         ->setParameters(new ArrayCollection([new Parameter('statusCrea', OrderStatus::STATUS_CREATED), new Parameter('from', $from), new Parameter('to', $to)]))
			         ->getQuery()
			         ->getScalarResult() as $row) {
			$tmp = DateTime::createFromFormat('Y-m-d H:i:s', $row['created']);

			if ($tmp) {
				$dates[(int) $row['order']] = $tmp;
			}
		}

		return $dates;
	}

	/**
	 * @return array<int, DateTime>
	 */
	public function getByStatus(DateTimeInterface $from, DateTimeInterface $to, string $status): array
	{
		$dates = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order, os.created')
			         ->from(OrderStatus::class, 'os')
			         ->andWhere('os.status = :status')
			         ->andWhere('os.created >= :from')
			         ->andWhere('os.created <= :to')
			         ->setParameters(new ArrayCollection([new Parameter('status', $status), new Parameter('from', $from), new Parameter('to', $to)]))->getQuery()->getScalarResult() as $row) {
			$tmp = DateTime::createFromFormat('Y-m-d H:i:s', $row['created']);

			if ($tmp) {
				$dates[(int) $row['order']] = $tmp;
			}
		}

		return $dates;
	}

	/**
	 * @return array<int, DateTime>
	 */
	public function getByPaid(DateTimeInterface $from, DateTimeInterface $to, bool $isPaid): array
	{
		$dates = [];
		foreach ($this->em->createQueryBuilder()->select('o.id, o.paid')
			         ->from(Order::class, 'o')
			         ->andWhere('o.paid >= :from')
			         ->andWhere('o.paid <= :to')
			         ->andWhere('o.isPaid = :isPaid')
			         ->setParameters(new ArrayCollection([new Parameter('from', $from), new Parameter('to', $to), new Parameter('isPaid', (int) $isPaid)]))->getQuery()->getScalarResult() as $row) {
			$tmp = DateTime::createFromFormat('Y-m-d H:i:s', $row['paid']);

			if ($tmp) {
				$dates[(int) $row['id']] = $tmp;
			}
		}

		return $dates;
	}

	public function getStockedInDate(DateTimeInterface $from, DateTimeInterface $to): array
	{
		if (!class_exists(EshopStockExtension::class)) {
			return [];
		}

		$supplyIds = [];
		$data      = [];
		foreach ($this->em->createQueryBuilder()->select('s.id')
			         ->from(Supply::class, 's')
			         ->where('s.dateSupply >= :from')
			         ->andWhere('s.dateSupply <= :to')
			         ->setParameters(new ArrayCollection([new Parameter('from', $from), new Parameter('to', $to)]))->getQuery()->getScalarResult() as $row) {
			$supplyIds[] = $row['id'];
		}

		foreach (array_chunk($supplyIds, 200) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(sp.product) as product, sp.purchasePrice')
				         ->from(SupplyProduct::class, 'sp')
				         ->where('sp.supply IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				if (!array_key_exists($row['product'], $data)) {
					$data[$row['product']] = [
						'count'         => 0,
						'purchasePrice' => 0,
					];
				}
				$data[$row['product']]['count']++;
				$data[$row['product']]['purchasePrice'] += (float) $row['purchasePrice'];
			}
		}


		return $data;
	}
}
