<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\Entities\SellerInvoiceConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Utils\Strings;
use Exception;
use Nette\Forms\IControl;
use Nette\Utils\ArrayHash;
use ReflectionException;

class InvoiceConfigForm extends BaseControl
{
	protected InvoiceConfigRepository $invoiceConfigRepository;

	protected ?InvoiceConfig $invoiceConfig;

	public function __construct(
		?int                    $id,
		InvoiceConfigRepository $invoiceConfigRepository,
		protected Invoices      $invoices,
		protected Sellers       $sellers,
	)
	{
		$this->invoiceConfig           = $id ? $invoiceConfigRepository->get($id) : null;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addCheckboxList(
			'sellers',
			'eshopOrders.invoiceConfigsGrid.sellers',
			$this->sellers->getOptionsForSelect(),
		)
			->setRequired($this->invoiceConfig ? true : false)
			->setDisabled(
				array_map(static fn(SellerInvoiceConfig $config,
				) => $config->seller->getId(), $this->invoiceConfigRepository->getUsed()),
			);

		$form->addInteger('maturity', 'eshopOrders.invoiceConfigForm.maturity')
			->setRequired()
			->addRule($form::MIN, null, 1);

		// Numerical series
		$form->addGroup('eshopOrders.invoiceConfigForm.numericalSeries.default.caption');
		$container = $form->addContainer('numericalSeries');
		$container->addText('prefix', 'eshopOrders.invoiceConfigForm.numericalSeries.default.prefix')
			->setRequired($this->invoiceConfig ? true : false)
			->addRule(static function(IControl $control,
			): bool { // If prefix contains month wildcards then it must also contain year wildcards
				$value = $control->getValue();

				return !(Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['month']) &&
					!Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['year']));
			}, 'eshopOrders.invoiceConfigForm.numericalSeries.errors.prefixValue');

		$container->addInteger('digitsCount', 'eshopOrders.invoiceConfigForm.numericalSeries.default.digitsCount')
			->setRequired($this->invoiceConfig ? true : false)
			->addRule($form::MIN, null, 1);

		$container->addInteger('startNumber', 'eshopOrders.invoiceConfigForm.numericalSeries.default.startNumber')
			->setRequired($this->invoiceConfig ? true : false);

		// Corrective numerical series
		$form->addGroup('eshopOrders.invoiceConfigForm.correctiveNumericalSeries.default.caption');
		$container = $form->addContainer('correctiveNumericalSeries');
		$container->addText('prefix', 'eshopOrders.invoiceConfigForm.correctiveNumericalSeries.default.prefix')
			->setRequired($this->invoiceConfig ? true : false)
			->addRule(static function(IControl $control,
			): bool { // If prefix contains month wildcards then it must also contain year wildcards
				$value = $control->getValue();

				return !(Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['month']) &&
					!Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['year']));
			}, 'eshopOrders.invoiceConfigForm.correctiveNumericalSeries.errors.prefixValue');

		$container->addInteger(
			'digitsCount',
			'eshopOrders.invoiceConfigForm.correctiveNumericalSeries.default.digitsCount',
		)
			->setRequired($this->invoiceConfig ? true : false)
			->addRule($form::MIN, null, 1);

		$container->addInteger(
			'startNumber',
			'eshopOrders.invoiceConfigForm.correctiveNumericalSeries.default.startNumber',
		)
			->setRequired($this->invoiceConfig ? true : false);

		$form->addHidden('id');

		$form->addSubmit('submit', 'default.save');

		$form->onAnchor[] = function(BaseForm $form) {

			// we cannot change numerical series if invoices count is bigger than 0
			if ($this->invoiceConfig && $this->invoices->getIvoicesCount($this->getSellersId()) > 0) {
				foreach (['numericalSeries', 'correctiveNumericalSeries'] as $v) {
					foreach (['prefix', 'digitsCount', 'startNumber'] as $v2) {
						$form[$v][$v2]->setDisabled();
					}
				}
			}
		};

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	/**
	 *
	 * @throws Exception
	 */
	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (!$values->id && (is_countable($values->sellers) ? count($values->sellers) : 0) === 0) {
			$form->addError('eshopOrders.invoiceConfigForm.numericalSeries.errors.sellersFilled');
		}

		$charsLength = 10;
		foreach (['numericalSeries', 'correctiveNumericalSeries'] as $v) {
			if ((Strings::length($values->{$v}->prefix) + $values->{$v}->digitsCount) > $charsLength) {
				// prefix + digits count char length must not exceed 10, becouse variable symbol is up to 10 chars
				$form->addError(
					$this->t(
						'eshopOrders.invoiceConfigForm.' . $v . '.errors.charsLength',
						null,
						['length' => $charsLength],
					),
				);
			} else if (Strings::length((string) $values->{$v}->startNumber) > $values->{$v}->digitsCount) {
				$form->addError('eshopOrders.invoiceConfigForm.' . $v . '.errors.maxStartNumber');
			}
		}

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	/**
	 *
	 * @throws Exception
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		if ($values->id) {
			$this->invoiceConfig = $this->invoiceConfigRepository->get((int) $values->id);
		}

		$numericalSeriesValues           = $values->numericalSeries;
		$correctiveNumericalSeriesValues = $values->correctiveNumericalSeries;

		if ($this->invoiceConfig === null) {
			$numericalSeries                      =
				new NumericalSeries(
					$numericalSeriesValues->prefix,
					$numericalSeriesValues->digitsCount,
					$numericalSeriesValues->startNumber
				);
			$correctiveNumericalSeries            =
				new NumericalSeries(
					$correctiveNumericalSeriesValues->prefix,
					$correctiveNumericalSeriesValues->digitsCount,
					$correctiveNumericalSeriesValues->startNumber
				);
			$this->invoiceConfig                  = new InvoiceConfig(
				(int) $values->maturity,
				$numericalSeries,
				$correctiveNumericalSeries
			);
			$this->invoiceConfig->numericalSeries = $numericalSeries;

			foreach ($values->sellers as $sellerId) {
				$seller = $this->sellers->get($sellerId);
				if (!$seller) {
					return;
				}
				$this->invoiceConfig->sellerInvoiceConfigs[] = new SellerInvoiceConfig($this->invoiceConfig, $seller);
			}
		} else {
			$defaultSellerInvoiceConfigs = $this->invoiceConfig->sellerInvoiceConfigs->toArray();
			$defaultSellers              = array_map(fn(SellerInvoiceConfig $config,
			) => $config->seller->getId(), $defaultSellerInvoiceConfigs);

			$newSellers = $values->seller;

			foreach (['numericalSeries', 'correctiveNumericalSeries'] as $v) {
				foreach (['prefix', 'digitsCount', 'startNumber'] as $v2) {
					$this->invoiceConfig->{$v}->{$v2} = $values->{$v}->{$v2};
				}
			}

			// remove
			foreach (array_diff($defaultSellers, $newSellers) as $sellerId) {
				foreach ($defaultSellerInvoiceConfigs as $invoiceSellerConfig) {
					if ($invoiceSellerConfig->seller->getId() === $sellerId) {
						$this->invoiceConfig->sellerInvoiceConfigs->removeElement($invoiceSellerConfig);
						$this->em->remove($invoiceSellerConfig);
						break;
					}
				}
			}

			// add
			foreach (array_diff($newSellers, $defaultSellers) as $sellerId) {
				$seller = $this->sellers->get($sellerId);
				if ($seller) {
					$this->invoiceConfig->sellerInvoiceConfigs->add(
						new SellerInvoiceConfig($this->invoiceConfig, $seller),
					);
				}
			}
		}

		$this->invoiceConfig->maturity = $values->maturity;

		$this->em->persist($this->invoiceConfig);
		$this->em->flush();
		$this->presenter->flashMessageSuccess('default.saved');
		$this->presenter->redrawControl('flashes');
	}

	/**
	 * @throws ReflectionException
	 */
	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @throws Exception
	 */
	public function setDefaults(): void
	{
		$defaultConfig = $this->invoiceConfig;

		$arr = array_map(static fn(SellerInvoiceConfig $config,
		) => $config->seller->getId(), $defaultConfig->sellerInvoiceConfigs->toArray());

		$d = [
			'maturity' => $defaultConfig->maturity,
			'sellers'  => $arr,
			'id'       => $defaultConfig->getId(),
		];

		foreach (['numericalSeries', 'correctiveNumericalSeries'] as $v) {
			foreach (['prefix', 'digitsCount', 'startNumber'] as $v2) {
				$d[$v][$v2] = $defaultConfig->{$v}->{$v2};
			}
		}

		$form = $this['form'];
		$form->setDefaults($d);
	}

	public function getSellersId(): array
	{
		return array_map(static fn(SellerInvoiceConfig $config,
		): int => $config->seller->getId(), $this->invoiceConfigRepository->getUsed());
	}

}
