<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Invoices;
use Nette\Utils\Html;
use Ublaboo\DataGrid\Exception\DataGridException;

class InvoicesGrid extends BaseControl
{
	/**
	 * InvoiceConfigForm constructor.
	 */
	public function __construct(protected Invoices $invoices, protected Price $priceFilter)
	{
	}

	/**
	 * @throws DataGridException
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->invoices->getQueryBuilder()
			->addSelect('iData, pay, sped, prods, disco, cust, o, oCurr')
			->leftJoin('i.invoiceData', 'iData')
			->leftJoin('iData.payment', 'pay')
			->leftJoin('iData.spedition', 'sped')
			->leftJoin('iData.customer', 'cust')
			->leftJoin('iData.products', 'prods')
			->leftJoin('iData.discounts', 'disco')
			->leftJoin('i.order', 'o')
			->leftJoin('o.currency', 'oCurr')
			->orderBy('i.createdAt', 'desc');
		$grid->setDataSource($qb);

		// columns
		$grid->addColumnText('ident', 'eshopOrders.invoicesGrid.ident')
			->setFilterText();
		$grid->addColumnLink(
			'order',
			'eshopOrders.invoicesGrid.order',
			':EshopOrders:Admin:Default:editOrder',
			'order.id',
		)
			->setRenderer(fn(Invoice $invoice) => $invoice->order
				? Html::el('a', [
					'href' => $this->getPresenter()
						->link(':EshopOrders:Admin:Default:editOrder', ['id' => $invoice->order->getId()]),
				])->setText($invoice->order->getId())
				: '')
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $val) {
				$qb->andWhere('o.id LIKE :order')
					->setParameter('order', "%{$val}%");
			});
		$grid->addColumnText('subscriber', 'eshopOrders.invoicesGrid.subscriber')->setRenderer(static function(
			Invoice $invoice,
		) {
			if ($invoice->isExternal()) {
				return '';
			}

			$customer = $invoice->invoiceData->getCustomer();

			return $customer->company ?? sprintf('%s %s', $customer->firstName, $customer->lastName);
		})
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $val) {
				$qb->andWhere(
					'cust.company LIKE :customer OR cust.firstName LIKE :customer OR cust.lastName LIKE :customer',
				)
					->setParameter('customer', "%{$val}%");
			});
		$grid->addColumnDateTime('fromDate', 'eshopOrders.invoicesGrid.fromDate')->setRenderer(static fn(
			Invoice $invoice,
		) => $invoice->createdAt->format('d. m. Y'))
			->setFilterDate('createdAt');
		$grid->addColumnDateTime('dueDate', 'eshopOrders.invoicesGrid.dueDate')
			->setFilterDate('createdAt');
		$grid->addColumnText('total', 'eshopOrders.invoicesGrid.totalPrice')->setRenderer(function(Invoice $invoice) {
			if ($invoice->isExternal()) {
				return '';
			}

			if ($invoice->order)
				return $invoice->order->renderPrice($invoice->invoiceData->getPrice(), true);

			return $this->priceFilter->format($invoice->invoiceData->getPrice());
		});

		// actions
		$grid->addAction('pdf', '')->setRenderer(function(Invoice $invoice) {
			if (!$invoice->order)
				return;

			return Html::el('a', ['href'  => $this->presenter->link('Orders:invoice', $invoice->order->getId()),
			                      'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
				->addHtml(Html::el('i', ['class' => 'fas fa-file-invoice']))
				->addHtml(Html::el('span')->setText($this->t('eshopOrders.invoicesGrid.pdf')));
		});

		if ($this->getPresenter()->getUser()->isAllowed('EshopOrders:Admin:Invoice', 'regenerateInvoices')) {
			$grid->addAction('regenerate', 'eshopOrders.invoicesGrid.actions.regenerate', 'regenerate!')
				->setRenderCondition(fn(Invoice $invoice) => !$invoice->isExternal())
				->setRenderer(function(Invoice $invoice) {
					if (!$invoice->order)
						return;

					return Html::el('a', ['href'  => $this->presenter->link('regenerate!', $invoice->getId()),
					                      'class' => 'btn btn-danger btn-xs ajax'])
						->addHtml(Html::el('i', ['class' => 'fas fa-sync']))
						->addHtml(Html::el('span')->setText($this->t('eshopOrders.invoicesGrid.actions.regenerate')));
				})
				->setConfirm('eshopOrders.invoice.reallyRegenerate');
		}

		return $grid;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

}
