<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Orders;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use Exception;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Tracy\Debugger;
use function str_starts_with;

class CorrectiveTaxDocumentForm extends BaseControl
{
	public function __construct(protected Order $order, protected Orders $orders)
	{
	}

	public function render(): void
	{
		$this->template->order = $this->order;
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		/** @var OrderItem $oi */
		foreach ($this->order->getOrderItems()->toArray() as $oi) {
			$container = $form->addContainer('prod_' . $oi->getId());
			$container->addHidden('oiId', $oi->getId());
			$container->addText('price')
				->setDefaultValue($oi->getPrice())
				->addCondition($form::FILLED)
				->addRule($form::FLOAT)
				->addRule($form::MIN, null, 0)
				->addRule($form::MAX, null, $oi->getPrice());
			$container->addInteger('quantity')
				->setDefaultValue($oi->getQuantity())
				->addCondition($form::FILLED)
				->addRule($form::MIN, null, 1)
				->addRule($form::MAX, null, $oi->getQuantity());
		}

		$orderDiscounts = $this->order->getOrderDiscounts()->toArray();
		usort($orderDiscounts, fn(
			OrderDiscount $o1,
			OrderDiscount $o2,
		) => $o1->getSortPosition() <=> $o2->getSortPosition());
		/** @var OrderDiscount $od */
		foreach ($orderDiscounts as $od) {
			$container = $form->addContainer('discount_' . $od->getId());
			$container->addHidden('odId', $od->getId());
			$container->addText('price')
				->setDefaultValue(abs($od->getPrice(true)))
				->addCondition($form::FILLED)
				->addRule($form::FLOAT)
				->addRule($form::MIN, null, 0);
		}

		$form->addText('paymentPrice', 'eshopOrders.correctiveTaxDocumentForm.paymentPrice')
			->setDefaultValue($this->order->getPayment()->getPrice())
			->setRequired()
			 ->addRule($form::FLOAT)
			->addRule($form::MIN, null, 0);
		$form->addText('speditionPrice', 'eshopOrders.correctiveTaxDocumentForm.speditionPrice')
			->setDefaultValue($this->order->getSpedition()->getPrice())
			->setRequired()
			 ->addRule($form::FLOAT)
			->addRule($form::MIN, null, 0);

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$vals        = $form->getHttpData();
		$hasProducts = Arrays::some(array_keys($vals), fn($inputKey) => str_starts_with((string) $inputKey, 'prod_'));
		if (!$hasProducts) {
			$form->addError('eshopOrders.correctiveTaxDocumentForm.errors.fillProducts');
			$this->redrawControl('formErrors');
			return;
		}

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->em->beginTransaction();

		try {
			$products = [];
			foreach ($form->getHttpData() as $k => $container) {
				if (!str_starts_with((string) $k, 'prod_')) {
					continue;
				}
				$productId            = (int) explode('_', (string) $k)[1];
				$products[$productId] = $container;
			}
			$orderDiscounts = [];
			foreach ($form->getHttpData() as $k => $container) {
				if (!str_starts_with((string) $k, 'discount_')) {
					continue;
				}
				$discountId                  = (int) explode('_', (string) $k)[1];
				$orderDiscounts[$discountId] = $container;
			}

			$this->orders->createCorrectiveTaxDocument(
				$this->order,
				$products,
				$orderDiscounts,
				$values->paymentPrice,
				$values->speditionPrice,
			);

			$this->em->commit();

			$this->presenter->flashMessageSuccess('eshopOrders.correctiveTaxDocumentForm.created');
		} catch (Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$this->presenter->flashMessageDanger('eshopOrders.correctiveTaxDocumentForm.savedError');
		}
	}

}
