<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\Event;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\AdminModule\Model\OrderGifts;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use Exception;
use Nette\Http\IResponse;

class OrderForm extends BaseControl
{
	public ?Order $order = null;

	public function __construct(
		protected Orders                                $ordersService,
		protected Speditions                            $speditionsService,
		protected Payments                              $paymentsService,
		protected CardsPaymentService                   $cardPaymentService,
		protected IOrderItemsGridFactory                $orderItemsGridFactory,
		protected IOrderStatusesGridFactory             $orderStatusesGridFactory,
		protected IIsPaidSwitchFactory                  $isPaidSwitchFactory,
		protected OrderGifts                            $orderGifts,
		protected Sellers                               $sellersService,
		protected PriceFilter                           $priceFilter,
		protected IEnableInvoiceGenerationSwitchFactory $enableInvoiceGenerationSwitchFactory,
		protected ICarrierHistoryGridFactory            $carrierHistoryGridFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->paymentsService = $this->paymentsService;
		$this->template->order           = $this->order;
		$this->template->seller          = $this->sellersService->getSellerForSite($this->order->site->getIdent());
		$this->template->showPaid        = EshopOrdersConfig::load('ordersGrid.showCardHistory') && (
				EshopOrdersConfig::load('showPaidFieldForAllOrders', false) === true
				|| $this->order->getPaymentIdent() === 'card');

		if (EshopOrdersConfig::load('ordersGrid.showCardHistory') && $this->order->getPaymentIdent() == 'card') {
			$this->template->paymentHistory = $this->cardPaymentService->getHistoryOfOrder($this->order->getId());
		}

		$this->template->render($this->getTemplateFile());
	}

	public function setOrder($id)
	{
		$this->order = $this->ordersService->get($id);

		if (!$this->order) {
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
		}
	}

	public function redrawStatuses(): void
	{
		$this['orderStatusesGrid']['grid']->reload();
	}

	public function handleRemoveGift($id): void
	{
		$presenter = $this->getPresenter();

		$orderGift = $this->orderGifts->get($id);
		$this->eventDispatcher->dispatch(new Event(['entity' => $orderGift]), 'eshopOrders.gifts.beforeDelete');

		$this->em->remove($orderGift);
		$this->em->flush();

		$presenter->flashMessageSuccess('default.removed');

		$this->redrawControl('gifts');
		$presenter->redrawControl('flashes');
	}

	public function handleValidate(string $type, string $val)
	{
		$presenter = $this->getPresenter();

		if ($type === 'idNumber') {
			$this->ordersService->validateIdNumber($this->order, (int) $val);
			$presenter->flashMessageSuccess('default.saved');
		} else if ($type === 'vatNumber') {
			$this->ordersService->validateVatNumber($this->order, (int) $val);
			$presenter->flashMessageSuccess('default.saved');
		} else {
			$presenter->flashMessageDanger('eshopOrders.order.validateNotFound');
		}

		$presenter->redirect('this');
	}

	public function handleChangePayment(int $id): void
	{
		$payment   = $this->paymentsService->get($id);
		$presenter = $this->presenter;

		if (!$presenter) {
			return;
		}

		if (!$payment) {
			$presenter->flashMessageDanger('eshopOrders.paymentNotFound');
		} else {
			try {
				$this->paymentsService->setOrderPayment(
					$this->order,
					$payment,
					$payment->getName(),
				);
				$presenter->flashMessageSuccess('eshopOrders.orderPaymentForm.edited');
			} catch (Exception) {
				$presenter->flashMessageDanger('default.error');
			}
		}

		$presenter->redrawControl('flashes');
		$this->redrawControl('payment');
		$this->redrawControl('spedition');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderItemsGrid()
	{
		$control = $this->orderItemsGridFactory->create($this->order->getId());

		return $control;
	}

	protected function createComponentOrderStatusesGrid()
	{
		$control = $this->orderStatusesGridFactory->create();
		if ($this->order) {
			$control->setOrder($this->order->getId());
		}

		return $control;
	}

	protected function createComponentCarrierHistoryGrid(): CarrierHistoryGrid
	{
		return $this->carrierHistoryGridFactory->create($this->order);
	}

	protected function createComponentIsPaidSwitch(): IsPaidSwitch
	{
		return $this->isPaidSwitchFactory->create($this->order);
	}

	protected function createComponentEnableInvoiceGenerationSwitch(): EnableInvoiceGenerationSwitch
	{
		return $this->enableInvoiceGenerationSwitchFactory->create($this->order);
	}
}
