<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\FormValidateEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use Exception;
use Nette\Application\UI\Form;
use Nette\Utils\ArrayHash;

class OrderItemForm extends BaseControl
{
	/** @var int @persistent */
	public $orderItemId;

	public ?OrderItem      $orderItem      = null;
	public ?OrderItemTexts $orderItemTexts = null;
	public ?Order          $order          = null;

	public function __construct(protected Orders $ordersService, protected OrderItems $orderItemsService)
	{
	}

	public function render(): void
	{
		$this->template->item = $this->orderItem;
		$this->template->render($this->getTemplateFile());
	}

	public function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->orderItemId && !$this->orderItem) {
			$this->setOrderItem($this->orderItemId);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	/**
	 * @param int $itemId
	 */
	public function handleDelete($itemId)
	{
		$presenter = $this->getPresenter();

		$this->orderItemsService->removeOrderItem((int) $itemId);

		$presenter->flashMessageSuccess('default.removed');
		$presenter->redrawControl('flashes');

		$presenter['orderForm']->redrawControl('items');
		$presenter['orderForm']->redrawControl('priceTotal');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addInteger('quantity', 'eshopOrders.default.quantity')
			->setRequired()
			->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('price', 'eshopOrders.default.pricePiece')
			->setRequired()
			->addRule(Form::FLOAT, 'Cena musí být číslo');

		if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
			$form->addText('vat', 'eshopOrders.default.vat')
				->setDescription('eshopOrders.orderItem.vatEditDescription')
				->setRequired();
		}

		if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
			$form->addTextArea('note', 'eshopOrders.orderItem.note')
				->setHtmlAttribute('rows', 3);
		}

		$form->addSaveCancelControl();
		$form->onValidate[] = function(BaseForm $form, ArrayHash $values) {
			$event                      = new FormValidateEvent(
				$form,
				$values,
				$this->getPresenterIfExists() ? $this->template : null,
				$this->getPresenterIfExists() ? $this->presenter : null
			);
			$event->custom['orderItem'] = $this->orderItem;
			$event->custom['order']     = $this->order;
			if ($this->getPresenterIfExists()) {
				$event->presenter = $this->presenter;
			}

			$this->eventDispatcher->dispatch($event, self::class . '::formValidate');

			if ($form->hasErrors()) {
				$this->redrawControl('form');
			}
		};
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->orderItem) {
				$orderItem    = $this->orderItem;
				$flashMessage = 'eshopOrders.itemForm.edited';
			} else {
				$orderItem    = new OrderItem(
					$this->em->getReference(Product::class, $values->product_id), $this->order
				);
				$flashMessage = 'eshopOrders.itemForm.added';
			}

			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);

			if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
				$orderItem->setVatRate((int) $values->vat);
			}

			if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
				$orderItem->setMoreDataValue('note', $values->note);
			}

			$this->orderItemsService->saveOrderItem($orderItem);

			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setOrder($orderId)
	{
		$this->order = $this->ordersService->get($orderId);
	}

	public function setOrderItem($orderItemId)
	{
		$this->orderItem      = $this->orderItemsService->get($orderItemId);
		$this->orderItemTexts = $this->orderItem->getOrderItemText('cs');
		$this->orderItemId    = $orderItemId;

		if ($this->orderItem) {
			$i = $this->orderItem;
			$d = [
				'quantity' => $i->getQuantity(),
				'price'    => $i->getPrice(),
				'note'     => $i->getMoreDataValue('note'),
			];

			if (EshopOrdersConfig::load('order.allowEditOrderItemVat')) {
				$d['vat'] = $i->getVatRate();
			}

			$this['form']->setDefaults($d);
		}
	}
}
