<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Payment;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use EshopOrders\Model\Payments;
use Exception;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class PaymentForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	public ?Payment $payment = null;

	public function __construct(
		protected Payments         $paymentsService,
		protected Currencies       $currencies,
		protected EshopOrdersCache $eshopOrdersCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$priceText = 'eshopOrders.default.priceSimple';
		if (EshopOrdersConfig::load('payments.priceIsWithoutVat'))
			$priceText = 'eshopOrders.default.priceSimpleWithoutVat';

		$form->addText('name', 'default.name')->setRequired()->setMaxLength(255);
		$form->addText('ident', 'eshopOrders.paymentForm.itemIdent')->setMaxLength(255);
		$form->addText('text', 'default.text')->setMaxLength(255);
		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addText('price', $priceText)
			->setHtmlType('number')
			->setHtmlAttribute('step', .01);

		$currencies = ['' => ''];
		foreach ($this->currencies->getActive() as $c)
			$currencies[$c->getCode()] = $c->getCode();
		$form->addSelect('currency', 'eshopOrders.payment.currency', $currencies);

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$form->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
		}

		$form->addInteger('availableFrom', 'eshopOrders.default.availableFrom');
		$form->addInteger('availableTo', 'eshopOrders.default.availableTo');
		$form->addFilesManager('image', 'default.image');

		$form->addBool('disableInGoogle', 'eshopOrders.payment.disableInGoogle')
			->setDefaultValue(0);
		if (EshopOrdersConfig::load('payments.allowPohodaName'))
			$form->addText('pohodaName', 'eshopOrders.payment.pohodaName')
				->setNullable();

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			PaymentForm::class . '::createForm',
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$payment      = $this->paymentsService->get($this->id);
				$flashMessage = 'eshopOrders.paymentForm.edited';
			} else {
				$payment      = new Payment();
				$flashMessage = 'eshopOrders.paymentForm.added';
			}

			$payment->setName($values->name);
			$payment->setIdent($values->ident);
			$payment->setText($values->text);
			$payment->isPublished = $values->isPublished;
			$payment->setPrice((float) $values->price);
			$payment->setAvailableFrom((int) $values->availableFrom);
			$payment->setAvailableTo((int) $values->availableTo);
			$payment->image           = $values->image;
			$payment->currency        = $values->currency ?: null;
			$payment->disableInGoogle = (int) $values->disableInGoogle;

			if (EshopOrdersConfig::load('allowFreeFrom')) {
				$payment->setFreeFrom(is_numeric($values->freeFrom) ? (int) $values->freeFrom : null);
			} else {
				$payment->setFreeFrom(null);
			}

			if (EshopOrdersConfig::load('payments.allowPohodaName'))
				$payment->pohodaName = $values->pohodaName;

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenterIfExists() ? $this->template : null,
				$this->getPresenterIfExists() ? $this->presenter : null
			);
			$event->custom['entity'] = $payment;
			$this->eventDispatcher->dispatch($event, PaymentForm::class . '::formSuccess');

			$this->em->persist($payment)->flush();
			$form->addCustomData('paymentId', $payment->getId());
			$this->presenter->flashMessageSuccess($flashMessage);

			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['payment', 'freeFrom'],
			]);

			return true;
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setPayment(int $id): void
	{
		$this->id      = $id;
		$this->payment = $p = $this->paymentsService->get($id);

		if (!$this->payment)
			throw new InvalidArgumentException();

		$d = [
			'name'            => $p->getName(),
			'ident'           => $p->getIdent(),
			'text'            => $p->getText(),
			'isPublished'     => $p->isPublished,
			'price'           => $p->getPrice(),
			'freeFrom'        => $p->getFreeFrom(),
			'availableFrom'   => $p->getAvailableFrom(),
			'availableTo'     => $p->getAvailableTo(),
			'image'           => $p->image,
			'currency'        => $p->currency,
			'disableInGoogle' => $p->disableInGoogle,
			'pohodaName'      => EshopOrdersConfig::load('payments.allowPohodaName') ? $p->pohodaName : null,
		];

		$this['form']->setDefaults($d);

		$this->eventDispatcher->dispatch(
			new SetFormDataEvent($this['form'], $this->payment),
			PaymentForm::class . '::setPayment',
		);
	}
}
