<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model\Statistics;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\DBAL\Logging\Middleware;
use Doctrine\ORM\Query;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use Nette\Utils\DateTime;
use Psr\Log\NullLogger;

class Overview
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var array */
	protected $cData;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
		$this->em->getConfiguration()->setMiddlewares([new Middleware(new NullLogger)]);
	}

	public function getOrdersCount(): array
	{
		$data = $this->getData()['ordersCount'];

		$result = [
			'count'   => $data['now'],
			'change'  => $data['now'] - $data['last'],
			'percent' => $data['last'] == 0 ? 0 : $data['now'] / ($data['last'] / 100) - 100,
		];

		return $result;
	}

	public function getOrdersSales(): array
	{
		$data = $this->getData()['sales'];

		$result = [
			'count'   => $data['now'],
			'change'  => $data['now'] - $data['last'],
			'percent' => $data['last'] == 0 ? 0 : $data['now'] / ($data['last'] / 100) - 100,
		];

		return $result;
	}

	protected function getData(): array
	{
		if ($this->cData === null) {
			$data      = [
				'ordersCount' => [
					'last' => 0,
					'now'  => 0,
				],
				'sales'       => [
					'last' => 0,
					'now'  => 0,
				],
			];
			$now       = (new DateTime())->setTime(0, 0, 0);
			$separator = (clone $now)->modify('-30 days');

			$created  = [];
			$canceled = [];
			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status, os.deleted, os.created')
				         ->andWhere('os.created >= :created')
				         ->setParameters([
					         'created' => (clone $now)->modify('-60 days'),
				         ])->groupBy('os.id')->getQuery()->getArrayResult() as $row) {
				if ($row['status'] == 'created')
					$created[$row['order']] = $row['created'];
				else if ($row['status'] == 'canceled' && $row['deleted'] != 1) {
					$canceled[$row['order']] = $row['created'];
				}
			}
			$ordersIds = array_diff_key($created, $canceled);

			foreach (array_chunk(array_keys($ordersIds), 600) as $chunk) {
				foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
					         ->addSelect('p, s, oi, oiSales, curr, od')
					         ->join('o.payment', 'p')
					         ->join('o.spedition', 's')
					         ->join('o.orderItems', 'oi')
					         ->leftJoin('oi.sales', 'oiSales')
					         ->leftJoin('o.currency', 'curr')
					         ->leftJoin('o.orderDiscounts', 'od')
					         ->where('o.id IN (' . implode(',', $chunk) . ')')
					         ->getQuery()
					         ->setHint(Query::HINT_FORCE_PARTIAL_LOAD, 1)
					         ->getResult() as $order) {
					/** @var Order $order */
					$created = $ordersIds[$order->getId()];
					if ($created > $separator) {
						$data['ordersCount']['now']++;
						$data['sales']['now'] += $order->getPriceItemsDiscount();
					} else {
						$data['ordersCount']['last']++;
						$data['sales']['last'] += $order->getPriceItemsDiscount();
					}
				}
			}

			$this->cData = $data;
		}

		return $this->cData;
	}
}
